# run_skeleton_task.py (Completed Executable Code)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills: pick, place, move, rotate, pull

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        # Example usage: positions = {'object_1': (x,y,z), ...}
        positions = get_object_positions()

        # === Exploration Phase: Identify Missing Predicate ===
        # Feedback: (holding gripper) is missing. We need to explore to determine if the robot is holding the gripper/object.
        # We'll attempt to pick an object and check if the predicate (holding gripper) is set in the environment.
        # This is a diagnostic/exploration step.

        # Find a pickable object and its position
        pickable_obj = None
        pickable_pos = None
        for obj_name, pos in positions.items():
            if "gripper" not in obj_name.lower() and "drawer" not in obj_name.lower():
                pickable_obj = obj_name
                pickable_pos = pos
                break
        if pickable_obj is None:
            # Fallback: just pick the first object
            for obj_name, pos in positions.items():
                pickable_obj = obj_name
                pickable_pos = pos
                break

        print(f"[Exploration] Attempting to pick object: {pickable_obj} at {pickable_pos}")
        try:
            obs, reward, done = pick(
                env,
                task,
                target_pos=np.array(pickable_pos),
                approach_distance=0.15,
                max_steps=100,
                threshold=0.01,
                approach_axis='z',
                timeout=10.0
            )
            # After pick, check if the predicate (holding gripper) is set
            # Since we don't have direct access to predicates, we print the observation for manual inspection
            print("[Exploration] Observation after pick:", obs)
            # If done, the task may have ended
            if done:
                print("[Exploration] Task ended after pick. Exiting exploration phase.")
                return
        except Exception as e:
            print(f"[Exploration] Exception during pick: {e}")

        # === Main Task Plan (Example) ===
        # Here, you would execute the oracle plan using only the available skills.
        # For demonstration, let's try to place the object back if pick succeeded.

        if pickable_obj is not None and pickable_pos is not None:
            # Find a place location (for example, offset in x direction)
            place_pos = np.array(pickable_pos) + np.array([0.2, 0, 0])
            print(f"[Task] Attempting to place object: {pickable_obj} at {place_pos}")
            try:
                obs, reward, done = place(
                    env,
                    task,
                    target_pos=place_pos,
                    approach_distance=0.15,
                    max_steps=100,
                    threshold=0.01,
                    approach_axis='z',
                    timeout=10.0
                )
                print("[Task] Observation after place:", obs)
                if done:
                    print("[Task] Task ended after place.")
                    return
            except Exception as e:
                print(f"[Task] Exception during place: {e}")

        # === Exploration Result ===
        print("[Exploration] If the predicate (holding gripper) is not set after pick, it is missing in the domain or not updated by the skill.")

        # === End of Task ===

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()