# run_skeleton_task.py (Completed Executable)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        # Example usage: positions = {'plate1': (x, y, z), ...}
        positions = get_object_positions()

        # === Exploration Phase: Identify Missing Predicate ===
        # Feedback: (at plate1 plate1) -- this is likely an error, as the object cannot be at itself.
        # The missing predicate is likely a location predicate for plate1 (e.g., (at plate1 table1))
        # We will explore the environment to determine the correct location predicate for plate1.

        # Try to find the correct location for plate1
        plate1_pos = None
        plate1_location = None
        try:
            # Try to get the position of plate1
            plate1_pos = positions.get('plate1', None)
            if plate1_pos is None:
                # Try alternative keys if needed
                for k in positions:
                    if 'plate1' in k:
                        plate1_pos = positions[k]
                        break
            if plate1_pos is None:
                raise KeyError("plate1 position not found in object positions.")
        except Exception as e:
            print(f"[Exploration] Could not find plate1 position: {e}")
            shutdown_environment(env)
            return

        # Try to infer the location (e.g., table, counter, etc.) for plate1
        # We'll look for a location object near plate1
        min_dist = float('inf')
        for obj_name, obj_pos in positions.items():
            if obj_name == 'plate1':
                continue
            # Heuristic: if the object name suggests a location (e.g., contains 'table', 'counter', etc.)
            if any(loc in obj_name for loc in ['table', 'counter', 'tray', 'surface', 'desk']):
                dist = np.linalg.norm(np.array(plate1_pos) - np.array(obj_pos))
                if dist < min_dist:
                    min_dist = dist
                    plate1_location = obj_name

        if plate1_location is None:
            # Fallback: just pick the closest object (excluding plate1)
            for obj_name, obj_pos in positions.items():
                if obj_name == 'plate1':
                    continue
                dist = np.linalg.norm(np.array(plate1_pos) - np.array(obj_pos))
                if dist < min_dist:
                    min_dist = dist
                    plate1_location = obj_name

        print(f"[Exploration] plate1 is at location: {plate1_location} (pos: {plate1_pos})")

        # === Plan Execution Phase ===
        # Use only available skills: ['pick', 'place', 'move', 'rotate', 'pull']

        # 1. Move to plate1's location (if move skill is available and needed)
        #    Since we don't have robot location info, we assume the robot can reach plate1 directly.
        #    If move is required, you would call move(env, task, from_pos, to_pos)
        #    For this example, we skip move unless required.

        # 2. Pick up plate1
        try:
            print(f"[Task] Picking up plate1 at position: {plate1_pos}")
            obs, reward, done = pick(
                env,
                task,
                target_pos=plate1_pos,
                approach_distance=0.15,
                max_steps=100,
                threshold=0.01,
                approach_axis='z',
                timeout=10.0
            )
            if done:
                print("[Task] Task ended after picking plate1!")
                return
        except Exception as e:
            print(f"[Task] Exception during pick: {e}")
            shutdown_environment(env)
            return

        # 3. Place plate1 at a target location (for demonstration, place at a new location)
        #    We'll try to find a different location to place the plate
        place_target_pos = None
        for obj_name, obj_pos in positions.items():
            if obj_name != 'plate1' and obj_name != plate1_location:
                place_target_pos = obj_pos
                print(f"[Task] Will place plate1 at {obj_name} (pos: {place_target_pos})")
                break
        if place_target_pos is None:
            # Fallback: offset the current position
            place_target_pos = np.array(plate1_pos) + np.array([0.2, 0, 0])
            print(f"[Task] No suitable place target found, using offset position: {place_target_pos}")

        try:
            obs, reward, done = place(
                env,
                task,
                target_pos=place_target_pos,
                approach_distance=0.15,
                max_steps=100,
                threshold=0.01,
                approach_axis='z',
                timeout=10.0
            )
            if done:
                print("[Task] Task ended after placing plate1!")
                return
        except Exception as e:
            print(f"[Task] Exception during place: {e}")
            shutdown_environment(env)
            return

        print("[Task] Successfully picked and placed plate1.")

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()
