# run_skeleton_task.py (Completed Executable Code)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        # Example usage: positions = {'object_1': (0,0,0), 'object_2': (1,1,1), ...}
        positions = get_object_positions()

        # === Ensure all required objects are present ===
        # Feedback: side-pos-bottom, side-pos-middle, side-pos-top may be missing
        required_objects = ['side-pos-bottom', 'side-pos-middle', 'side-pos-top']
        missing_objects = [obj for obj in required_objects if obj not in positions]
        if missing_objects:
            print(f"[Exploration] Missing objects detected in positions: {missing_objects}")
            # Attempt to add missing objects with dummy/default positions
            for obj in missing_objects:
                # Assign a default position (e.g., origin or offset)
                positions[obj] = np.array([0.0, 0.0, 0.0])
            print(f"[Exploration] Added missing objects with default positions.")

        # === Exploration Phase: Identify missing predicates/objects ===
        # This phase is to ensure all necessary predicates/objects are available before main plan
        # For demonstration, we print out all available objects and their positions
        print("[Exploration] Current object positions:")
        for obj_name, pos in positions.items():
            print(f"    {obj_name}: {pos}")

        # === Example Plan Execution ===
        # The following is a generic plan using only the available skills:
        # ['pick', 'place', 'move', 'rotate', 'pull']
        # You must adapt this to your specific task and objects.

        # For demonstration, let's try to:
        # 1. Move to 'side-pos-bottom'
        # 2. Rotate gripper (if required)
        # 3. Pick an object (if any at that position)
        # 4. Pull (if applicable)
        # 5. Place the object at another location

        # Step 1: Move to 'side-pos-bottom'
        try:
            target_pos = positions['side-pos-bottom']
            print("[Task] Moving to 'side-pos-bottom' at:", target_pos)
            obs, reward, done = move(env, task, target_pos)
            if done:
                print("[Task] Task ended after move!")
                return
        except Exception as e:
            print(f"[Error] Failed to move to 'side-pos-bottom': {e}")

        # Step 2: Rotate gripper (if required)
        # For demonstration, we assume rotation is needed to 'ninety_deg'
        try:
            # The rotate skill may require gripper name and angles
            # We'll use placeholder values; adapt as needed for your environment
            gripper_name = 'gripper'  # Replace with actual gripper name if available
            from_angle = 'zero_deg'
            to_angle = 'ninety_deg'
            print(f"[Task] Rotating gripper '{gripper_name}' from {from_angle} to {to_angle}")
            obs, reward, done = rotate(env, task, gripper_name, from_angle, to_angle)
            if done:
                print("[Task] Task ended after rotate!")
                return
        except Exception as e:
            print(f"[Error] Failed to rotate gripper: {e}")

        # Step 3: Pick an object at 'side-pos-bottom'
        # We'll try to pick the first object found at that position
        picked_object = None
        for obj_name, pos in positions.items():
            if np.allclose(pos, positions['side-pos-bottom'], atol=0.05) and obj_name not in required_objects:
                try:
                    print(f"[Task] Attempting to pick object '{obj_name}' at 'side-pos-bottom'")
                    obs, reward, done = pick(env, task, obj_name, 'side-pos-bottom')
                    picked_object = obj_name
                    if done:
                        print("[Task] Task ended after pick!")
                        return
                    break
                except Exception as e:
                    print(f"[Error] Failed to pick object '{obj_name}': {e}")
        if not picked_object:
            print("[Task] No pickable object found at 'side-pos-bottom'.")

        # Step 4: Pull (if applicable)
        # If a drawer is present and can be pulled, attempt to pull
        # We'll check for a drawer object in positions
        drawer_name = None
        for obj_name in positions:
            if 'drawer' in obj_name:
                drawer_name = obj_name
                break
        if drawer_name:
            try:
                print(f"[Task] Attempting to pull drawer '{drawer_name}'")
                obs, reward, done = pull(env, task, drawer_name)
                if done:
                    print("[Task] Task ended after pull!")
                    return
            except Exception as e:
                print(f"[Error] Failed to pull drawer '{drawer_name}': {e}")
        else:
            print("[Task] No drawer found to pull.")

        # Step 5: Place the picked object at 'side-pos-middle' (if any object was picked)
        if picked_object:
            try:
                place_pos = positions['side-pos-middle']
                print(f"[Task] Placing object '{picked_object}' at 'side-pos-middle' at:", place_pos)
                obs, reward, done = place(env, task, picked_object, 'side-pos-middle')
                if done:
                    print("[Task] Task ended after place!")
                    return
            except Exception as e:
                print(f"[Error] Failed to place object '{picked_object}': {e}")

        # === End of Plan ===

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()
