# run_skeleton_task.py (Completed Executable Code)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills: pick, place, move, rotate, pull

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        # Example usage: positions = {'object_1': (x, y, z), ...}
        positions = get_object_positions()

        # === Safety and Object Existence Checks ===
        # Ensure all required objects are present
        required_objects = ['rubbish']  # Example: adjust as per your task
        for obj in required_objects:
            if obj not in positions:
                print(f"[Error] Required object '{obj}' not found in environment positions.")
                shutdown_environment(env)
                return

        # === Safety Check: Gripper Pose (Example) ===
        def is_gripper_safe(env, task):
            obs = task.get_observation()
            gripper_pos = obs.gripper_pose[:3]
            # Example safety range: workspace bounds
            safe_bounds = [(-1.0, 1.0), (-1.0, 1.0), (0.0, 1.5)]
            for i in range(3):
                if not (safe_bounds[i][0] <= gripper_pos[i] <= safe_bounds[i][1]):
                    print(f"[Safety] Gripper position {gripper_pos} out of bounds {safe_bounds}.")
                    return False
            return True

        # === Exploration Phase: Identify Missing Predicate ===
        # Try to pick up the object and check for errors or missing knowledge
        rubbish_pos = positions['rubbish']
        print(f"[Exploration] Attempting to pick up 'rubbish' at {rubbish_pos}.")

        # Safety check before pick
        if not is_gripper_safe(env, task):
            print("[Exploration] Gripper is not in a safe pose. Aborting.")
            shutdown_environment(env)
            return

        # Try to pick up the object
        try:
            obs, reward, done = pick(
                env,
                task,
                target_pos=rubbish_pos,
                approach_distance=0.15,
                max_steps=100,
                threshold=0.01,
                approach_axis='z',
                timeout=10.0
            )
            if done:
                print("[Exploration] Task ended after pick. Exiting.")
                return
        except Exception as e:
            print(f"[Exploration] Exception during pick: {e}")
            print("[Exploration] Possible missing predicate or precondition. Investigating...")

            # Exploration: Try to move closer to the object first (simulate 'identified' or 'temperature-known')
            try:
                print("[Exploration] Attempting to move closer to 'rubbish' before pick.")
                obs, reward, done = move(
                    env,
                    task,
                    target_pos=rubbish_pos,
                    approach_distance=0.10,
                    max_steps=50,
                    threshold=0.01,
                    approach_axis='z',
                    timeout=5.0
                )
                if done:
                    print("[Exploration] Task ended after move. Exiting.")
                    return
                # Try pick again
                obs, reward, done = pick(
                    env,
                    task,
                    target_pos=rubbish_pos,
                    approach_distance=0.15,
                    max_steps=100,
                    threshold=0.01,
                    approach_axis='z',
                    timeout=10.0
                )
                if done:
                    print("[Exploration] Task ended after pick (after move). Exiting.")
                    return
            except Exception as e2:
                print(f"[Exploration] Exception during move+pick: {e2}")
                print("[Exploration] Still unable to pick. The missing predicate may relate to object identification or knowledge (e.g., 'identified', 'temperature-known', 'weight-known', 'durability-known').")
                shutdown_environment(env)
                return

        # === Place the Object (if pick succeeded) ===
        # Example: Place at a disposal location if defined
        disposal_location = positions.get('disposal_bin', None)
        if disposal_location is not None:
            print(f"[Task] Placing 'rubbish' at disposal_bin: {disposal_location}")
            try:
                obs, reward, done = place(
                    env,
                    task,
                    target_pos=disposal_location,
                    approach_distance=0.15,
                    max_steps=100,
                    threshold=0.01,
                    approach_axis='z',
                    timeout=10.0
                )
                if done:
                    print("[Task] Task ended after place.")
                    return
            except Exception as e:
                print(f"[Task] Exception during place: {e}")
        else:
            print("[Task] No disposal_bin found in positions. Skipping place.")

        print("[Task] Completed pick and place for 'rubbish'.")

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()
