# run_skeleton_task.py (Completed Executable Code)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        # Example usage: positions = {'tomato_1': (x1,y1,z1), 'tomato_2': (x2,y2,z2), ...}
        try:
            positions = get_object_positions()
        except Exception as e:
            print("[Error] Failed to get object positions:", e)
            shutdown_environment(env)
            return

        # === Update object list to match environment ===
        # Remove any objects not present in the environment
        object_names = list(positions.keys())
        print("[Info] Objects in environment:", object_names)

        # === Exploration Phase: Identify Missing Predicate ===
        # The feedback and exploration domain suggest that the robot may need to "identify" or "sense" properties of objects
        # such as weight, durability, or lock status before manipulation.
        # We'll attempt to pick each object and check for errors, to see if a predicate (e.g., weight-known) is missing.

        # For demonstration, let's try to pick each tomato and place it on the plate.
        # We'll assume the plate is named 'plate' in the positions dict.
        # If not, we skip the place step.

        plate_name = None
        for name in object_names:
            if 'plate' in name:
                plate_name = name
                break

        # If no plate found, just print a warning
        if plate_name is None:
            print("[Warning] No plate found in object positions. Will only pick objects.")

        # For each tomato (or pickable object), try to pick and place
        for obj_name in object_names:
            if obj_name == plate_name:
                continue  # Don't try to pick the plate itself

            target_pos = positions[obj_name]
            print(f"[Task] Attempting to pick object: {obj_name} at {target_pos}")

            # === Check if object is within reach ===
            obs = task.get_observation()
            gripper_pos = obs.gripper_pose[:3]
            dist = np.linalg.norm(np.array(target_pos) - np.array(gripper_pos))
            max_reach = 0.5  # Example threshold, adjust as needed
            if dist > max_reach:
                print(f"[Error] Object {obj_name} is out of reach (distance: {dist:.3f}). Skipping.")
                continue

            # === Check gripper orientation (z-axis down) ===
            # For simplicity, assume approach_axis='z' is correct for all objects
            approach_axis = 'z'

            # === Try to pick the object ===
            try:
                obs, reward, done = pick(
                    env,
                    task,
                    target_pos=target_pos,
                    approach_distance=0.15,
                    max_steps=100,
                    threshold=0.01,
                    approach_axis=approach_axis,
                    timeout=10.0
                )
            except Exception as e:
                print(f"[Error] Exception during pick of {obj_name}: {e}")
                continue

            if done:
                print(f"[Task] Task ended after picking {obj_name}!")
                return

            # === If plate exists, place the object on the plate ===
            if plate_name is not None:
                plate_pos = positions[plate_name]
                print(f"[Task] Placing {obj_name} on plate at {plate_pos}")
                try:
                    obs, reward, done = place(
                        env,
                        task,
                        target_pos=plate_pos,
                        approach_distance=0.15,
                        max_steps=100,
                        threshold=0.01,
                        approach_axis='z',
                        timeout=10.0
                    )
                except Exception as e:
                    print(f"[Error] Exception during place of {obj_name} on plate: {e}")
                    continue

                if done:
                    print(f"[Task] Task ended after placing {obj_name} on plate!")
                    return

        # === Exploration: Try to pull drawers if present ===
        # If there are drawers, try to pull them using the pull skill
        for obj_name in object_names:
            if 'drawer' in obj_name or 'handle' in obj_name:
                print(f"[Exploration] Attempting to pull {obj_name}")
                # For pull, we need to know the gripper is holding the drawer handle
                # This may require a sequence: move gripper to handle, pick-drawer, then pull
                # Here, we only attempt pull if the skill is available and the object is present
                try:
                    obs, reward, done = pull(
                        env,
                        task,
                        target_pos=positions[obj_name],
                        approach_distance=0.10,
                        max_steps=100,
                        threshold=0.01,
                        approach_axis='z',
                        timeout=10.0
                    )
                except Exception as e:
                    print(f"[Error] Exception during pull of {obj_name}: {e}")
                    continue

                if done:
                    print(f"[Task] Task ended after pulling {obj_name}!")
                    return

        print("[Task] All pick-and-place (and pull) attempts completed.")

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()
