# run_skeleton_task.py (Completed with Exploration Phase for Missing Predicate)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills: move, pick, place, rotate, pull

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation, with exploration for missing predicates.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        # Example usage: positions = {'object_1': (0,0,0), 'object_2': (1,1,1), ...}
        positions = get_object_positions()

        # === Exploration Phase: Identify Missing Predicate ===
        # The feedback indicates a missing predicate is causing planning to fail.
        # We will attempt to explore the environment using available skills to discover which predicate is missing.
        # The exploration domain suggests predicates like identified, temperature-known, weight-known, durability-known, lock-known.
        # We will try to perform actions and observe if any information is missing or if any action fails due to missing knowledge.

        # For demonstration, we will attempt to:
        # 1. Move to each object location (if possible)
        # 2. Try to pick each object (if possible)
        # 3. Try to pull (if possible)
        # 4. Observe if any action fails or if any information is missing

        # Note: The actual object and location names depend on your environment.
        # We'll iterate over all objects in positions and attempt actions.

        # List of available skills
        available_skills = ['pick', 'place', 'move', 'rotate', 'pull']

        # Track which predicates might be missing
        missing_predicates = set()

        # Attempt to explore each object
        for obj_name, obj_pos in positions.items():
            print(f"[Exploration] Attempting to interact with object: {obj_name} at position {obj_pos}")

            # 1. Try to move to the object's position
            try:
                print(f"[Exploration] Moving to {obj_name} ...")
                obs, reward, done = move(env, task, target_pos=np.array(obj_pos))
                if done:
                    print(f"[Exploration] Task ended after move to {obj_name}!")
                    break
            except Exception as e:
                print(f"[Exploration] Move to {obj_name} failed: {e}")
                missing_predicates.add('robot-at or at')

            # 2. Try to pick the object
            try:
                print(f"[Exploration] Attempting to pick {obj_name} ...")
                obs, reward, done = pick(
                    env,
                    task,
                    target_pos=np.array(obj_pos),
                    approach_distance=0.15,
                    max_steps=100,
                    threshold=0.01,
                    approach_axis='z',
                    timeout=10.0
                )
                if done:
                    print(f"[Exploration] Task ended after picking {obj_name}!")
                    break
            except Exception as e:
                print(f"[Exploration] Pick {obj_name} failed: {e}")
                # If pick fails, it may be due to missing 'identified', 'weight-known', or 'durability-known'
                missing_predicates.add('identified/weight-known/durability-known')

            # 3. Try to pull (if the object is a drawer or has a handle)
            # This is a guess; in a real scenario, you'd check object type
            if 'drawer' in obj_name or 'handle' in obj_name:
                try:
                    print(f"[Exploration] Attempting to pull {obj_name} ...")
                    obs, reward, done = pull(env, task, obj_name)
                    if done:
                        print(f"[Exploration] Task ended after pulling {obj_name}!")
                        break
                except Exception as e:
                    print(f"[Exploration] Pull {obj_name} failed: {e}")
                    # If pull fails, it may be due to missing 'lock-known'
                    missing_predicates.add('lock-known')

        # After exploration, print out which predicates might be missing
        if missing_predicates:
            print("[Exploration] Potential missing predicates detected:")
            for pred in missing_predicates:
                print(f"    - {pred}")
        else:
            print("[Exploration] No missing predicates detected during exploration.")

        # === (Optional) Main Task Plan Execution ===
        # If you have an oracle plan, you would execute it here using the available skills.
        # For now, the focus is on exploration to find missing predicates.

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()