# run_skeleton_task.py (Completed with Exploration Phase)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills: move, pick, place, rotate, pull

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation, with exploration for missing predicates.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        # Example usage: positions = {'object_1': (x, y, z), ...}
        positions = get_object_positions()

        # === Exploration Phase: Try to Discover Missing Predicate ===
        # The feedback suggests a missing predicate is blocking planning.
        # We will attempt to execute each available skill on each object/location
        # and observe if any action fails due to a missing predicate.

        available_skills = ['pick', 'place', 'move', 'rotate', 'pull']
        exploration_results = {}

        # For demonstration, we will try to pick and move each object to each location
        # and catch any exceptions or errors that may indicate a missing predicate.

        print("=== [Exploration] Starting exploration to find missing predicate ===")
        for obj_name, obj_pos in positions.items():
            # Try pick
            try:
                print(f"[Exploration] Trying to pick {obj_name} at {obj_pos}")
                obs, reward, done = pick(
                    env,
                    task,
                    target_pos=obj_pos,
                    approach_distance=0.15,
                    max_steps=100,
                    threshold=0.01,
                    approach_axis='z',
                    timeout=10.0
                )
                exploration_results[(obj_name, 'pick')] = 'success'
                if done:
                    print(f"[Exploration] Task ended after picking {obj_name}!")
                    break
            except Exception as e:
                print(f"[Exploration] Pick failed for {obj_name}: {e}")
                exploration_results[(obj_name, 'pick')] = f'fail: {e}'

            # Try move (move gripper to object position)
            try:
                print(f"[Exploration] Trying to move to {obj_name} at {obj_pos}")
                obs, reward, done = move(
                    env,
                    task,
                    target_pos=obj_pos,
                    max_steps=100,
                    threshold=0.01,
                    timeout=10.0
                )
                exploration_results[(obj_name, 'move')] = 'success'
                if done:
                    print(f"[Exploration] Task ended after moving to {obj_name}!")
                    break
            except Exception as e:
                print(f"[Exploration] Move failed for {obj_name}: {e}")
                exploration_results[(obj_name, 'move')] = f'fail: {e}'

            # Try place (if pick succeeded)
            try:
                # For place, we need a target location. Use the first available location different from obj_pos.
                for loc_name, loc_pos in positions.items():
                    if loc_name != obj_name:
                        print(f"[Exploration] Trying to place {obj_name} at {loc_pos}")
                        obs, reward, done = place(
                            env,
                            task,
                            target_pos=loc_pos,
                            approach_distance=0.15,
                            max_steps=100,
                            threshold=0.01,
                            approach_axis='z',
                            timeout=10.0
                        )
                        exploration_results[(obj_name, 'place', loc_name)] = 'success'
                        if done:
                            print(f"[Exploration] Task ended after placing {obj_name} at {loc_name}!")
                            break
                        break  # Only try one place per object
            except Exception as e:
                print(f"[Exploration] Place failed for {obj_name}: {e}")
                exploration_results[(obj_name, 'place')] = f'fail: {e}'

        # Try rotate and pull if there are drawers or similar objects
        for obj_name, obj_pos in positions.items():
            # Try rotate (if applicable)
            try:
                print(f"[Exploration] Trying to rotate {obj_name}")
                # For rotate, we need gripper, from_angle, to_angle. Use dummy values if not available.
                obs, reward, done = rotate(
                    env,
                    task,
                    gripper_name='gripper',  # Replace with actual gripper name if available
                    from_angle='zero_deg',   # Replace with actual angle if available
                    to_angle='ninety_deg',   # Replace with actual angle if available
                    max_steps=100,
                    threshold=0.01,
                    timeout=10.0
                )
                exploration_results[(obj_name, 'rotate')] = 'success'
                if done:
                    print(f"[Exploration] Task ended after rotating {obj_name}!")
                    break
            except Exception as e:
                print(f"[Exploration] Rotate failed for {obj_name}: {e}")
                exploration_results[(obj_name, 'rotate')] = f'fail: {e}'

            # Try pull (if applicable)
            try:
                print(f"[Exploration] Trying to pull {obj_name}")
                obs, reward, done = pull(
                    env,
                    task,
                    object_name=obj_name,
                    max_steps=100,
                    threshold=0.01,
                    timeout=10.0
                )
                exploration_results[(obj_name, 'pull')] = 'success'
                if done:
                    print(f"[Exploration] Task ended after pulling {obj_name}!")
                    break
            except Exception as e:
                print(f"[Exploration] Pull failed for {obj_name}: {e}")
                exploration_results[(obj_name, 'pull')] = f'fail: {e}'

        print("=== [Exploration] Exploration results ===")
        for k, v in exploration_results.items():
            print(f"  {k}: {v}")

        print("=== [Exploration] Finished. If any action failed due to a missing predicate, check the error messages above. ===")

        # === End of Exploration Phase ===

        # TODO: After identifying the missing predicate, update the domain/problem as needed and rerun the planner.

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()