# run_skeleton_task.py (Completed Executable Code)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        # Example usage: positions = {'object_1': (x,y,z), ...}
        positions = get_object_positions()

        # === Exploration Phase: Identify Missing Predicate ===
        # Feedback: (holding gripper) is missing. We need to explore to determine what predicate is missing.
        # We'll attempt to pick an object and check if the state reflects the expected predicate.

        # For demonstration, pick the first object in positions
        object_names = list(positions.keys())
        if not object_names:
            print("[Exploration] No objects found in the environment.")
            return

        test_obj = object_names[0]
        test_obj_pos = positions[test_obj]
        print(f"[Exploration] Attempting to pick object: {test_obj} at {test_obj_pos}")

        try:
            obs, reward, done = pick(
                env,
                task,
                target_pos=np.array(test_obj_pos),
                approach_distance=0.15,
                max_steps=100,
                threshold=0.01,
                approach_axis='z',
                timeout=10.0
            )
        except Exception as e:
            print(f"[Exploration] Exception during pick: {e}")
            shutdown_environment(env)
            return

        # After pick, check if the predicate (holding gripper) is reflected in the state
        # Since we don't have direct access to the PDDL state, we print a message for manual inspection
        print("[Exploration] Pick action executed. Please check if the predicate (holding gripper) is now true in the environment state.")
        print("[Exploration] If not, (holding gripper) is the missing predicate.")

        # === End of Exploration Phase ===

        # === Main Task Plan (Example) ===
        # Here you would continue with the oracle plan using only the available skills:
        # ['pick', 'place', 'move', 'rotate', 'pull']
        # For demonstration, we show a simple pick and place if more than one object exists.

        if len(object_names) > 1:
            target_obj = object_names[1]
            target_obj_pos = positions[target_obj]
            print(f"[Task] Attempting to place {test_obj} at {target_obj}'s position: {target_obj_pos}")
            try:
                obs, reward, done = place(
                    env,
                    task,
                    target_pos=np.array(target_obj_pos),
                    approach_distance=0.15,
                    max_steps=100,
                    threshold=0.01,
                    approach_axis='z',
                    timeout=10.0
                )
            except Exception as e:
                print(f"[Task] Exception during place: {e}")
                shutdown_environment(env)
                return
            if done:
                print("[Task] Task ended after place!")
                return
        else:
            print("[Task] Only one object found; skipping place action.")

        # You can add more skill invocations (move, rotate, pull) as needed for your oracle plan.

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()
