# run_task_open_and_dispose.py

import numpy as np
from scipy.spatial.transform import Rotation as R

from env import setup_environment, shutdown_environment
from video import init_video_writers, recording_step, recording_get_observation
from object_positions import get_object_positions

# === Import every predefined skill ===
from skill_code import rotate, move, pick, pull, place


def safe_step(step_fn, *args, **kwargs):
    """Utility wrapper that calls a skill function and gracefully stops
    execution if the task signals done=True or an exception is thrown."""
    try:
        obs, reward, done = step_fn(*args, **kwargs)
        if done:
            print("[Task] Environment returned done=True.")
        return obs, reward, done
    except Exception as e:
        print(f"[Task] Exception during skill execution: {e}")
        raise


def run_task():
    """Concrete task script that:
       1) Opens (pulls) the bottom drawer.
       2) Picks ‘item3’ (our rubbish surrogate) from the table.
       3) Places it in the bin."""
    print("===== Starting Open‑Drawer & Dispose Task =====")

    # ------------------------
    # Environment setup
    # ------------------------
    env, task = setup_environment()
    try:
        descriptions, obs = task.reset()

        # Optional video recording helpers
        init_video_writers(obs)
        task.step = recording_step(task.step)
        task.get_observation = recording_get_observation(task.get_observation)

        # ------------------------
        # Retrieve all useful object positions
        # ------------------------
        positions = get_object_positions()

        # Required keys (assert early for clearer error messages)
        required_keys = ["bottom_side_pos", "bottom_anchor_pos",
                         "item3", "bin"]
        for key in required_keys:
            if key not in positions:
                raise KeyError(f"[Task] Missing '{key}' in object_positions.")

        bottom_side_pos   = np.asarray(positions["bottom_side_pos"])
        bottom_anchor_pos = np.asarray(positions["bottom_anchor_pos"])
        rubbish_pos       = np.asarray(positions["item3"])       # treat item3 as rubbish
        bin_pos           = np.asarray(positions["bin"])

        # ------------------------
        # Step‑by‑Step Oracle Plan
        # ------------------------
        done = False

        # 1) rotate – turn gripper 90° about Z so it is ‘side‑on’ to the drawer
        print("\n[Step 1] rotate → 90°")
        target_quat = R.from_euler('z', 90, degrees=True).as_quat()  # xyzw
        obs, reward, done = safe_step(
            rotate, env, task,
            target_quat=target_quat,
            max_steps=120, threshold=0.05, timeout=10.0
        )
        if done: return

        # 2) move – go to the pre‑recorded side position of the bottom drawer
        print("\n[Step 2] move → bottom_side_pos:", bottom_side_pos)
        obs, reward, done = safe_step(
            move, env, task,
            target_pos=bottom_side_pos,
            max_steps=120, threshold=0.01, timeout=10.0
        )
        if done: return

        # 3) move – from side to anchor (front/handle) position
        print("\n[Step 3] move → bottom_anchor_pos:", bottom_anchor_pos)
        obs, reward, done = safe_step(
            move, env, task,
            target_pos=bottom_anchor_pos,
            max_steps=120, threshold=0.01, timeout=10.0
        )
        if done: return

        # 4) pick – close on drawer handle (bottom_anchor_pos)
        print("\n[Step 4] pick drawer @ anchor")
        obs, reward, done = safe_step(
            pick, env, task,
            target_pos=bottom_anchor_pos,
            approach_distance=0.04,
            max_steps=120, threshold=0.01,
            approach_axis='y',  # approach directly towards drawer face
            timeout=10.0
        )
        if done: return

        # 5) pull – pull the drawer out by 20 cm along +X
        print("\n[Step 5] pull drawer")
        obs, reward, done = safe_step(
            pull, env, task,
            pull_distance=0.20,
            pull_axis='x',     # positive X (adjust if drawer axis differs)
            max_steps=120, threshold=0.01,
            timeout=10.0
        )
        if done: return

        # 6) pick – release drawer (gripper opens internally) and pick the rubbish
        print("\n[Step 6] pick rubbish (item3) @", rubbish_pos)
        obs, reward, done = safe_step(
            pick, env, task,
            target_pos=rubbish_pos,
            approach_distance=0.15,
            max_steps=150, threshold=0.01,
            approach_axis='z',
            timeout=12.0
        )
        if done: return

        # 7) place – drop rubbish into the bin
        print("\n[Step 7] place into bin @", bin_pos)
        obs, reward, done = safe_step(
            place, env, task,
            target_pos=bin_pos,
            approach_distance=0.20,
            max_steps=150, threshold=0.01,
            approach_axis='z',
            timeout=12.0
        )

        # ------------------------
        # Task finished
        # ------------------------
        if done:
            print("[Task] Task signalled completion with reward:", reward)
        else:
            print("[Task] Plan executed. Final reward:", reward)

    finally:
        # Always shut down the environment even if something goes wrong
        shutdown_environment(env)
        print("===== End of Task =====")


if __name__ == "__main__":
    run_task()
