import numpy as np
from scipy.spatial.transform import Rotation as R

from env import setup_environment, shutdown_environment
from object_positions import get_object_positions
from video import init_video_writers, recording_step, recording_get_observation

# === Import ALL predefined skills ===
from skill_code import rotate, move, pick, pull, place


def run_skeleton_task():
    """
    High‑level controller that
      1) opens the middle drawer (thereby “unlocking” the cabinet),
      2) throws the trash away,
      3) leaves everything else untouched.
    The concrete sequence strictly follows the 9‑step oracle plan required
    by the specification.
    """
    print("===== Starting Skeleton Task =====")

    # ------------------------------------------------------------------
    #  Environment initialisation
    # ------------------------------------------------------------------
    env, task = setup_environment()
    try:
        descriptions, obs = task.reset()

        # Optionally start video recording
        init_video_writers(obs)
        task.step = recording_step(task.step)
        task.get_observation = recording_get_observation(task.get_observation)

        # ------------------------------------------------------------------
        #  Retrieve all relevant object positions from the scene
        # ------------------------------------------------------------------
        positions = get_object_positions()

        # Drawer handle / side / anchor points
        side_mid      = np.array(positions['middle_side_pos'])
        anchor_mid    = np.array(positions['middle_anchor_pos'])

        # Trash (we assume item3 is the rubbish)
        trash_name    = 'item3'
        trash_pos     = np.array(positions[trash_name])

        # Bin position
        bin_pos       = np.array(positions['bin'])

        # Convenience offsets
        Z_UP          = np.array([0.0, 0.0, 0.15])    # 15 cm above a point
        Z_SAFE        = np.array([0.0, 0.0, 0.25])    # higher “safe” height

        # ------------------------------------------------------------------
        #  Oracle Plan – 9 Steps
        # ------------------------------------------------------------------

        # STEP‑1  rotate gripper from 0 deg ➜ 90 deg around Z
        target_quat = R.from_euler('z', 90, degrees=True).as_quat()
        print("\n[Plan‑1] rotate → 90° about Z")
        obs, reward, done = rotate(env, task, target_quat)
        if done:
            print("[Early Termination] after rotate.")
            return

        # STEP‑2  move to drawer SIDE position (slightly above)
        print("\n[Plan‑2] move → drawer side position")
        obs, reward, done = move(env, task, side_mid + Z_UP)
        if done:
            print("[Early Termination] after move‑2.")
            return

        # STEP‑3  move to drawer ANCHOR position (slightly above)
        print("\n[Plan‑3] move → drawer anchor position (approach)")
        obs, reward, done = move(env, task, anchor_mid + Z_UP)
        if done:
            print("[Early Termination] after move‑3.")
            return

        # STEP‑4  pick the drawer handle at anchor position
        print("\n[Plan‑4] pick → grasp drawer handle")
        obs, reward, done = pick(env, task, anchor_mid)
        if done:
            print("[Early Termination] after pick‑4.")
            return

        # STEP‑5  pull the drawer straight out (along +X by 0.20 m)
        print("\n[Plan‑5] pull → open drawer")
        obs, reward, done = pull(env, task, pull_distance=0.20, pull_axis='x')
        if done:
            print("[Early Termination] after pull‑5.")
            return

        # STEP‑6  move up to a safe height above the table
        print("\n[Plan‑6] move → safe hover position above table")
        obs, reward, done = move(env, task, trash_pos + Z_SAFE)
        if done:
            print("[Early Termination] after move‑6.")
            return

        # STEP‑7  pick the trash object
        print("\n[Plan‑7] pick → grasp trash (item3)")
        obs, reward, done = pick(env, task, trash_pos)
        if done:
            print("[Early Termination] after pick‑7.")
            return

        # STEP‑8  move to bin (hover)
        print("\n[Plan‑8] move → bin hover position")
        obs, reward, done = move(env, task, bin_pos + Z_UP)
        if done:
            print("[Early Termination] after move‑8.")
            return

        # STEP‑9  place trash in bin
        print("\n[Plan‑9] place → drop trash into bin")
        obs, reward, done = place(env, task, bin_pos)
        if done:
            print("[Task Completed] Successfully opened drawer & disposed trash!")
        else:
            print("[Task Incomplete] done flag is False after final place.")

    except Exception as e:
        print("Exception during task execution:", e)

    finally:
        shutdown_environment(env)
        print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()
