# run_combined_drawer_disposal_task.py

import numpy as np
from scipy.spatial.transform import Rotation as R

from env import setup_environment, shutdown_environment

# Import ALL pre‑implemented skills exactly as provided
from skill_code import move, rotate, pick, pull, place

# Helpers for video recording (stay exactly as in skeleton)
from video import init_video_writers, recording_step, recording_get_observation

# Helper that gives us the Cartesian positions of every interesting object
from object_positions import get_object_positions


def _safe_call(skill_fn, *args, **kwargs):
    """
    Generic wrapper that calls <skill_fn> and prints a nice message if the
    skill terminates the RLBench episode (done == True) or throws an error.
    """
    try:
        obs, reward, done = skill_fn(*args, **kwargs)
        if done:
            print(f"[Task] Episode finished early inside {skill_fn.__name__} (done=True).")
        return obs, reward, done
    except Exception as e:
        print(f"[Task] Exception inside {skill_fn.__name__}: {e}")
        raise


def run_combined_drawer_disposal_task():
    """
    Full task:
      1) Move the gripper to the *side* position of the bottom drawer
      2) Rotate gripper 90° so that fingers point toward drawer handle
      3) Move to the drawer's *anchor* (handle) position
      4) Grasp the handle
      5) Pull the drawer open
      6) Move above the rubbish (item3) lying on the table
      7) Pick up the rubbish
      8) Move over the trash bin
      9) Drop the rubbish in the bin
    These 9 physical steps map 1‑to‑1 to the 9 “specification” steps.
    """
    print("\n====================  START TASK  ====================")

    # -------------------- ENV SETUP -------------------- #
    env, task = setup_environment()
    try:
        descriptions, obs = task.reset()          # RLBench reset
        init_video_writers(obs)                   # Optional – video capture
        # Wrap the task for recording every step
        task.step = recording_step(task.step)
        task.get_observation = recording_get_observation(
            task.get_observation
        )

        # ---------------- OBJECT POSITIONS -------------- #
        positions = get_object_positions()

        def p(name):
            """Small helper to convert dictionary entry to numpy array."""
            return np.array(positions[name], dtype=np.float32)

        # All named positions from the object list
        bottom_side     = p("bottom_side_pos")
        bottom_anchor   = p("bottom_anchor_pos")
        item_rubbish    = p("item3")           # Treat item3 as the 'rubbish'
        bin_pos         = p("bin")

        # A little vertical offset so we approach from above and avoid collision
        Z_OFFSET = np.array([0.0, 0.0, 0.10], dtype=np.float32)

        # ------------- STEP‑BY‑STEP EXECUTION ------------ #
        done_global = False   # Track if episode ended early

        # STEP‑1  (move)  – go to “side” position above drawer
        if not done_global:
            tgt = bottom_side + Z_OFFSET
            print(f"\n[Step‑1] move → bottom_side_pos + offset  @  {tgt}")
            obs, reward, done_global = _safe_call(
                move, env, task, target_pos=tgt
            )

        # STEP‑2  (rotate) – rotate from 0° to 90° around global Z
        if not done_global:
            rot_quat = R.from_euler("z", 90, degrees=True).as_quat()
            print(f"\n[Step‑2] rotate → 90° about Z  (quat={rot_quat})")
            obs, reward, done_global = _safe_call(
                rotate, env, task, target_quat=rot_quat
            )

        # STEP‑3  (move)  – move down to the drawer anchor (handle)
        if not done_global:
            tgt = bottom_anchor
            print(f"\n[Step‑3] move → bottom_anchor_pos  @  {tgt}")
            obs, reward, done_global = _safe_call(
                move, env, task, target_pos=tgt
            )

        # STEP‑4  (pick)  – grasp the drawer handle
        if not done_global:
            print(f"\n[Step‑4] pick → drawer handle at {bottom_anchor}")
            obs, reward, done_global = _safe_call(
                pick,
                env,
                task,
                target_pos=bottom_anchor,
                approach_distance=0.05,
                approach_axis="-z"
            )

        # STEP‑5  (pull)  – pull the drawer open along +X
        if not done_global:
            print("\n[Step‑5] pull → open drawer by 0.18 m along +X")
            obs, reward, done_global = _safe_call(
                pull,
                env,
                task,
                pull_distance=0.18,
                pull_axis="x"
            )

        # STEP‑6  (move)  – move to rubbish (above it)
        if not done_global:
            tgt = item_rubbish + Z_OFFSET
            print(f"\n[Step‑6] move → above rubbish (item3)  @  {tgt}")
            obs, reward, done_global = _safe_call(
                move, env, task, target_pos=tgt
            )

        # STEP‑7  (pick)  – pick the rubbish item
        if not done_global:
            print(f"\n[Step‑7] pick → rubbish at {item_rubbish}")
            obs, reward, done_global = _safe_call(
                pick,
                env,
                task,
                target_pos=item_rubbish,
                approach_distance=0.12,
                approach_axis="-z"
            )

        # STEP‑8  (move)  – move over the trash bin
        if not done_global:
            tgt = bin_pos + Z_OFFSET
            print(f"\n[Step‑8] move → above trash bin  @  {tgt}")
            obs, reward, done_global = _safe_call(
                move, env, task, target_pos=tgt
            )

        # STEP‑9  (place) – drop rubbish into the bin
        if not done_global:
            print(f"\n[Step‑9] place → inside bin at {bin_pos}")
            obs, reward, done_global = _safe_call(
                place,
                env,
                task,
                target_pos=bin_pos,
                approach_distance=0.12,
                approach_axis="-z"
            )

        # -------------------- DONE ---------------------- #
        if done_global:
            print("\n[Task] Episode reported done before final step.")
        else:
            print("\n[Task] All steps executed. Check reward / success signal.")

    finally:
        # Environment clean‑up no matter what
        shutdown_environment(env)
        print("====================  END TASK  =====================")


if __name__ == "__main__":
    run_combined_drawer_disposal_task()
