import numpy as np
from scipy.spatial.transform import Rotation as R

from env import setup_environment, shutdown_environment
from skill_code import rotate, move, pick, pull, place
from video import init_video_writers, recording_step, recording_get_observation
from object_positions import get_object_positions


def run_skeleton_task():
    '''Execute the oracle plan (rotate → move → pick → pull → pick → place).'''
    print("===== Starting Skeleton Task =====")

    # ---------------------------------------------------------------------
    # 1)  Environment setup
    # ---------------------------------------------------------------------
    env, task = setup_environment()
    try:
        descriptions, obs = task.reset()

        # (Optional) video recording helpers
        init_video_writers(obs)
        task.step = recording_step(task.step)
        task.get_observation = recording_get_observation(task.get_observation)

        # -----------------------------------------------------------------
        # 2)  Retrieve all useful object positions
        # -----------------------------------------------------------------
        positions = get_object_positions()  # dict: name -> np.ndarray([x, y, z])

        bottom_side_pos = np.asarray(positions['bottom_side_pos'])
        bottom_anchor_pos = np.asarray(positions['bottom_anchor_pos'])

        rubbish_pos = np.asarray(positions['rubbish'])
        bin_pos = np.asarray(positions['bin'])

        # -----------------------------------------------------------------
        # 3)  Execute the oracle plan
        # -----------------------------------------------------------------
        done = False

        # ----- Step‑1  : rotate gripper from zero_deg → ninety_deg -----
        ninety_quat_xyzw = R.from_euler('z', 90, degrees=True).as_quat()
        obs, reward, done = rotate(env, task, target_quat=ninety_quat_xyzw)
        if done:
            print("[Task] Episode finished unexpectedly after rotate.")
            return

        # ----- Step‑2  : move to side‑pos of bottom drawer -----
        obs, reward, done = move(env, task, target_pos=bottom_side_pos)
        if done:
            print("[Task] Episode finished unexpectedly after move‑to‑side.")
            return

        # ----- Step‑3  : move to anchor‑pos of bottom drawer -----
        obs, reward, done = move(env, task, target_pos=bottom_anchor_pos)
        if done:
            print("[Task] Episode finished unexpectedly after move‑to‑anchor.")
            return

        # ----- Step‑4  : pick (grasp) the drawer handle at anchor‑pos -----
        #    Approach from +y direction so the gripper comes from outside the drawer
        obs, reward, done = pick(
            env, task,
            target_pos=bottom_anchor_pos,
            approach_axis='y'
        )
        if done:
            print("[Task] Episode finished unexpectedly after pick‑drawer.")
            return

        # ----- Step‑5  : pull the drawer outward along x‑axis -----
        #    Decide pulling axis (+x or −x) based on workspace geometry
        pull_axis = 'x'
        if bottom_anchor_pos[0] > 0:   # if the handle is already on +x side, pull towards +x
            pull_axis = 'x'
        else:
            pull_axis = '-x'
        obs, reward, done = pull(
            env, task,
            pull_distance=0.20,   # 20 cm pull distance
            pull_axis=pull_axis
        )
        if done:
            print("[Task] Episode finished unexpectedly after pull.")
            return

        # -----------------------------------------------------------------
        # Drawer is now open – next handle table rubbish
        # -----------------------------------------------------------------

        # ----- Step‑6  : pick the rubbish from the table -----
        obs, reward, done = pick(
            env, task,
            target_pos=rubbish_pos,
            approach_axis='z'
        )
        if done:
            print("[Task] Episode finished unexpectedly after pick‑rubbish.")
            return

        # ----- Step‑7  : place the rubbish in the bin -----
        obs, reward, done = place(
            env, task,
            target_pos=bin_pos,
            approach_axis='z'
        )
        if done:
            print("[Task] Task completed (rubbish discarded).  Reward:", reward)
        else:
            print("[Task] Plan executed, but environment signalled done=False.")

    except Exception as e:
        print("[Task] Exception occurred:", str(e))
        raise
    finally:
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()