# run_skeleton_task.py
#
# Task‑specific implementation that follows the oracle plan contained in
# the “Specification” section of the prompt.  We rely strictly on the
# predefined skills imported from skill_code (rotate, move, pick, pull,
# place).  No new low‑level skills are added — only high‑level control
# flow that executes the seven ordered steps.
#
# NOTE:
#   • Object names come from the supplied object list.
#   • World‑space poses are obtained via get_object_positions().
#   • If an object pose cannot be found we raise a RuntimeError so the
#     problem surfaces immediately instead of silently failing.

import numpy as np
from scipy.spatial.transform import Rotation as R

from env import setup_environment, shutdown_environment
from video import init_video_writers, recording_step, recording_get_observation
from object_positions import get_object_positions

# Import ALL predefined skills exactly as they exist
from skill_code import rotate, move, pick, pull, place


# -------------------------------------------------------------
# Helper utilities
# -------------------------------------------------------------
def _quat_from_euler(roll=0.0, pitch=0.0, yaw=0.0):
    """
    Convenience wrapper that converts Euler angles (deg) into a
    normalised quaternion in (x, y, z, w) order.
    """
    quat = R.from_euler('xyz', [np.deg2rad(roll),
                                np.deg2rad(pitch),
                                np.deg2rad(yaw)]).as_quat()
    # R.as_quat returns (x, y, z, w) already
    return quat / np.linalg.norm(quat)


def _lookup_position(positions_dict, key):
    """
    Fetch an (x,y,z) numpy array for `key` or raise a clear error.
    """
    if key not in positions_dict:
        raise RuntimeError(f"[run_task] Could not find object pose for key: '{key}'")
    return np.asarray(positions_dict[key], dtype=np.float32)


# -------------------------------------------------------------
# Oracle‑plan execution
# -------------------------------------------------------------
def run_skeleton_task():
    print("========== [Task] START ==========")

    # ---------------- Environment setup ----------------
    env, task = setup_environment()
    try:
        descriptions, obs = task.reset()

        # Optional video capture
        init_video_writers(obs)
        task.step = recording_step(task.step)
        task.get_observation = recording_get_observation(task.get_observation)

        # Query all known object positions once (they update every tick
        # internally, but static queries are good enough for this demo).
        positions = get_object_positions()

        # Resolve all PDDL‑style symbols to simulator waypoints
        middle_side_pos     = _lookup_position(positions, 'middle_side_pos')
        middle_anchor_pos   = _lookup_position(positions, 'middle_anchor_pos')
        rubbish_pos         = _lookup_position(positions, 'rubbish')
        bin_pos             = _lookup_position(positions, 'bin')

        # ---------------- Step‑by‑step plan ----------------
        done = False    # RLBench “done” flag
        reward = 0.0

        # STEP‑1  rotate(gripper, zero_deg -> ninety_deg)
        print("\n[Step‑1] rotate gripper 0° → 90° about Z")
        ninety_quat = _quat_from_euler(0.0, 0.0, 90.0)     # yaw 90°
        obs, reward, done = rotate(env, task, target_quat=ninety_quat)
        if done:
            print("[Task] Episode finished during step 1.")
            return

        # STEP‑2  move‑to‑side(gripper, middle, nowhere‑pos → side‑pos‑middle)
        print("\n[Step‑2] move gripper to middle_side_pos")
        obs, reward, done = move(env, task, target_pos=middle_side_pos)
        if done:
            print("[Task] Episode finished during step 2.")
            return

        # STEP‑3  move‑to‑anchor(gripper, middle, side‑pos‑middle → anchor‑pos‑middle)
        print("\n[Step‑3] move gripper to middle_anchor_pos")
        obs, reward, done = move(env, task, target_pos=middle_anchor_pos)
        if done:
            print("[Task] Episode finished during step 3.")
            return

        # STEP‑4  pick‑drawer(gripper, middle, anchor‑pos‑middle)
        print("\n[Step‑4] pick drawer handle at anchor‑pos‑middle")
        obs, reward, done = pick(env, task, target_pos=middle_anchor_pos,
                                 approach_distance=0.12, approach_axis='-z')
        if done:
            print("[Task] Episode finished during step 4.")
            return

        # STEP‑5  pull(gripper, middle) → open the drawer along +X by 0.12 m
        print("\n[Step‑5] pull drawer outward along +X")
        obs, reward, done = pull(env, task, pull_distance=0.12, pull_axis='x')
        if done:
            print("[Task] Episode finished during step 5.")
            return

        # Release the drawer by opening the gripper slightly so that the
        # subsequent rubbish pick can succeed (handempty in symbolic world).
        print("\n[Cleanup] Releasing drawer handle (open gripper)")
        # The easiest way: execute a zero‑movement place at current pose
        # so that the gripper opens.  We reuse “place” with the same
        # end‑effector position to simply open the fingers.
        current_pos = obs.gripper_pose[:3]
        obs, reward, done = place(env, task, target_pos=current_pos,
                                  approach_distance=0.0)   # no approach, stay
        if done:
            print("[Task] Episode finished while releasing drawer.")
            return

        # STEP‑6  pick(rubbish, table)
        print("\n[Step‑6] pick rubbish object from the table")
        obs, reward, done = pick(env, task, target_pos=rubbish_pos,
                                 approach_distance=0.15, approach_axis='-z')
        if done:
            print("[Task] Episode finished during step 6.")
            return

        # STEP‑7  place(rubbish, bin)
        print("\n[Step‑7] place rubbish into bin")
        obs, reward, done = place(env, task, target_pos=bin_pos,
                                  approach_distance=0.15, approach_axis='-z')
        if done:
            print(f"[Task] Completed! Final reward: {reward}")
        else:
            print("[Task] Plan executed, but the environment did not signal 'done'.")
    finally:
        shutdown_environment(env)
        print("========== [Task] END ==========")


if __name__ == "__main__":
    run_skeleton_task()