# run_skeleton_task.py
#
# Generic execution script that follows the oracle plan described in the
# specification.  It assumes that all low‑level skills (move, rotate, pick,
# pull, place) are already implemented and imported from skill_code.
#
# The high‑level sequence carried out here is:
#
#   1) Move the gripper to the side position of the (bottom) drawer.
#   2) Rotate the gripper by 90 ° so that it can grasp the drawer handle.
#   3) Move to the anchor position (handle) of the drawer.
#   4) Pick / grasp the drawer handle.
#   5) Pull the drawer open.
#   6) Pick the rubbish that lies on the table.
#   7) Place the rubbish into the bin.
#
# The script terminates immediately if the task environment signals “done”.

import numpy as np
from scipy.spatial.transform import Rotation as R
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

# Import all low‑level skills that were supplied separately
from skill_code import move, rotate, pick, pull, place

# Video helpers (optional, but harmless if video module is a stub)
from video import (
    init_video_writers,
    recording_step,
    recording_get_observation,
)

# Helper that returns a dict  {object_name: np.ndarray([x,y,z])}
from object_positions import get_object_positions


def get_quaternion_from_euler(euler_xyz_deg):
    """
    Utility to convert an (x,y,z) Euler angle in degrees into a xyzw quaternion
    that the rotate() skill expects.
    """
    return R.from_euler("xyz", np.deg2rad(euler_xyz_deg)).as_quat()


def run_skeleton_task():
    print("===== Starting Skeleton Task =====")

    # ------------------------------------------------------------------
    #  Environment initialisation
    # ------------------------------------------------------------------
    env, task = setup_environment()
    try:
        # RLBench style reset (returns textual descriptions & observation)
        descriptions, obs = task.reset()

        # Optional video recording (works even if dummy implementation)
        init_video_writers(obs)
        task.step = recording_step(task.step)
        task.get_observation = recording_get_observation(task.get_observation)

        # ------------------------------------------------------------------
        #  Fetch all relevant object positions from the helper module
        # ------------------------------------------------------------------
        positions = get_object_positions()

        # ---- Drawer related positions --------------------------------------------------
        side_pos  = positions.get("bottom_side_pos")
        anchor_pos = positions.get("bottom_anchor_pos")

        if side_pos is None or anchor_pos is None:
            raise RuntimeError(
                "Could not retrieve drawer way‑points "
                "(bottom_side_pos / bottom_anchor_pos) from object_positions."
            )

        # ---- Rubbish & bin positions ---------------------------------------------------
        #  The PDDL uses “rubbish”, but the actual RLBench asset is called “item3”.
        rubbish_pos = positions.get("item3")
        bin_pos     = positions.get("bin")

        if rubbish_pos is None or bin_pos is None:
            raise RuntimeError(
                "Missing 'item3' (rubbish) or 'bin' position in object_positions."
            )

        # ------------------------------------------------------------------
        #  Oracle plan execution (Specification steps 1 … 7)
        # ------------------------------------------------------------------

        # STEP‑1 : move → side position next to drawer
        print("\n[Step‑1] move to side_pos_bottom")
        obs, reward, done = move(
            env, task, target_pos=side_pos, max_steps=150, threshold=0.01
        )
        if done:
            print("[Task] Environment signalled done after step‑1.")
            return

        # STEP‑2 : rotate → 90 ° about z‑axis (gripper points towards drawer)
        print("\n[Step‑2] rotate to ninety_deg")
        ninety_deg_quat = get_quaternion_from_euler([0, 0, 90])
        obs, reward, done = rotate(
            env, task, target_quat=ninety_deg_quat, max_steps=150, threshold=0.05
        )
        if done:
            print("[Task] Environment signalled done after step‑2.")
            return

        # STEP‑3 : move → anchor (handle) position
        print("\n[Step‑3] move to anchor_pos_bottom")
        obs, reward, done = move(
            env, task, target_pos=anchor_pos, max_steps=150, threshold=0.01
        )
        if done:
            print("[Task] Environment signalled done after step‑3.")
            return

        # STEP‑4 : pick → grasp drawer handle
        print("\n[Step‑4] pick (drawer handle at anchor_pos_bottom)")
        obs, reward, done = pick(
            env,
            task,
            target_pos=anchor_pos,
            approach_distance=0.10,
            approach_axis="z",
            timeout=15.0,
        )
        if done:
            print("[Task] Environment signalled done after step‑4.")
            return

        # STEP‑5 : pull → open the drawer (pull along +x by 0.20 m)
        print("\n[Step‑5] pull drawer")
        obs, reward, done = pull(
            env,
            task,
            pull_distance=0.20,
            pull_axis="x",
            max_steps=150,
            threshold=0.01,
        )
        if done:
            print("[Task] Environment signalled done after step‑5.")
            return

        # OPTIONAL: move upwards slightly before moving to the table objects
        current_pos = obs.gripper_pose[:3]
        safe_height = current_pos + np.array([0.0, 0.0, 0.10])
        obs, reward, done = move(env, task, target_pos=safe_height)
        if done:
            return

        # STEP‑6 : pick → rubbish (item3) lying on the table
        print("\n[Step‑6] pick rubbish (item3)")
        obs, reward, done = pick(
            env,
            task,
            target_pos=rubbish_pos,
            approach_distance=0.15,
            approach_axis="z",
            timeout=15.0,
        )
        if done:
            print("[Task] Environment signalled done after step‑6.")
            return

        # STEP‑7 : place → drop rubbish into the bin
        print("\n[Step‑7] place rubbish into bin")
        obs, reward, done = place(
            env,
            task,
            target_pos=bin_pos,
            approach_distance=0.15,
            approach_axis="z",
            timeout=15.0,
        )

        # ------------------------------------------------------------------
        #  Final status
        # ------------------------------------------------------------------
        if done:
            print("\n===== Task finished successfully! Reward:", reward, "=====")
        else:
            print("\n===== Task sequence completed, but 'done' was False. =====")

    finally:
        shutdown_environment(env)
        print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()
