import numpy as np
from scipy.spatial.transform import Rotation as R

# ------------------------------------------------------------------
# run_oracle_task.py
#
# This script executes the oracle plan required for:
#   “Unlock the cabinet by opening an accessible drawer, then throw
#    away the trash while leaving any other objects alone.”
#
# The exact high‑level steps are fixed by the Specification section:
#
#   1) rotate
#   2) move
#   3) move
#   4) pick      (grab drawer handle)
#   5) pull      (open drawer)
#   6) pick      (pick up trash)
#   7) place     (throw it away)
#
# ------------------------------------------------------------------

import traceback
from env import setup_environment, shutdown_environment

# keep skeleton imports unchanged
from skill_code import *                      # noqa: F401,F403
from video import init_video_writers, recording_step, recording_get_observation
from object_positions import get_object_positions


def run_oracle_task():
    print("=====  ORACLE TASK – START  =====")
    env, task = setup_environment()

    try:
        # ----------------------------------------------------------
        # Reset the task / obtain initial observation
        # ----------------------------------------------------------
        descriptions, obs = task.reset()

        # (Optional) video recording
        init_video_writers(obs)
        task.step = recording_step(task.step)
        task.get_observation = recording_get_observation(task.get_observation)

        # ----------------------------------------------------------
        # Retrieve all relevant positions from the helper utility
        # ----------------------------------------------------------
        positions = get_object_positions()

        # Drawer related
        bottom_side_pos   = np.asarray(positions.get('bottom_side_pos',   [0., 0., 0.]))
        bottom_anchor_pos = np.asarray(positions.get('bottom_anchor_pos', [0., 0., 0.]))

        # Trash & disposal
        trash_pos = np.asarray(positions.get('item3', [0., 0., 0.]))   # assume item3 == trash
        bin_pos   = np.asarray(positions.get('bin',   [0., 0., 0.]))

        # ----------------------------------------------------------
        # STEP‑1  rotate  (rotate gripper to 90° about Z)
        # ----------------------------------------------------------
        target_quat = R.from_euler('xyz', [0, 0, 90], degrees=True).as_quat()
        print("\n[Plan‑1] rotate → ninety deg")
        obs, reward, done = rotate(env, task, target_quat)
        if done:
            print("Task ended early during rotate!"); return

        # ----------------------------------------------------------
        # STEP‑2  move  (to drawer side position)
        # ----------------------------------------------------------
        print("\n[Plan‑2] move → bottom_side_pos =", bottom_side_pos)
        obs, reward, done = move(env, task, bottom_side_pos)
        if done:
            print("Task ended early during move‑2!"); return

        # ----------------------------------------------------------
        # STEP‑3  move  (to drawer anchor/handle position)
        # ----------------------------------------------------------
        print("\n[Plan‑3] move → bottom_anchor_pos =", bottom_anchor_pos)
        obs, reward, done = move(env, task, bottom_anchor_pos)
        if done:
            print("Task ended early during move‑3!"); return

        # ----------------------------------------------------------
        # STEP‑4  pick  (grab the drawer handle)
        # ----------------------------------------------------------
        print("\n[Plan‑4] pick → drawer handle at anchor_pos")
        obs, reward, done = pick(env, task, bottom_anchor_pos, approach_axis='z')
        if done:
            print("Task ended early during pick‑drawer!"); return

        # ----------------------------------------------------------
        # STEP‑5  pull  (open the drawer)
        # ----------------------------------------------------------
        print("\n[Plan‑5] pull → open drawer")
        obs, reward, done = pull(env, task,
                                 pull_distance=0.12,   # ~12 cm outward
                                 pull_axis='x')
        if done:
            print("Task ended early during pull!"); return

        # ----------------------------------------------------------
        # STEP‑6  pick  (pick up trash item3)
        # ----------------------------------------------------------
        print("\n[Plan‑6] pick → trash (item3) at", trash_pos)
        obs, reward, done = pick(env, task, trash_pos, approach_axis='z')
        if done:
            print("Task ended early during pick‑trash!"); return

        # ----------------------------------------------------------
        # STEP‑7  place  (drop trash into bin)
        # ----------------------------------------------------------
        print("\n[Plan‑7] place → bin at", bin_pos)
        obs, reward, done = place(env, task, bin_pos, approach_axis='z')
        if done:
            print("=====  ORACLE TASK – SUCCESS  =====")
        else:
            # Some tasks in RLBench never set done=True; treat as success
            print("=====  ORACLE TASK – FINISHED (done flag not set)  =====")

    except Exception as exc:
        print("=====  ORACLE TASK – ERROR  =====")
        traceback.print_exc()
    finally:
        shutdown_environment(env)


if __name__ == "__main__":
    run_oracle_task()