import numpy as np
from scipy.spatial.transform import Rotation as R

from env import setup_environment, shutdown_environment
from video import init_video_writers, recording_step, recording_get_observation
from object_positions import get_object_positions
from skill_code import *      # rotate, move, pull, pick, place


def run_skeleton_task():
    """
    Oracle plan (taken from the specification):

        1) rotate   : turn the gripper 90 deg so that it meets the drawer‑side constraint
        2) move     : go to the bottom drawer’s side‑handle position
        3) move     : move onto the bottom drawer’s anchor position (handle centre)
        4) pull     : pull the handle to open the (un‑locked) bottom drawer
        5) move     : travel to the trash object lying on the table
        6) pick     : grasp the trash
        7) move     : travel to the bin
        8) place    : drop the trash into the bin
    """
    print("===== Starting Skeleton Task =====")

    # ------------------------------------------------------------------
    #  Environment initialisation
    # ------------------------------------------------------------------
    env, task = setup_environment()
    try:
        descriptions, obs = task.reset()

        # (optional) start video recording
        init_video_writers(obs)
        task.step = recording_step(task.step)
        task.get_observation = recording_get_observation(task.get_observation)

        # ------------------------------------------------------------------
        #  Gather all object positions we will need
        # ------------------------------------------------------------------
        positions = get_object_positions()          # dictionary[str -> np.ndarray]

        # Resolve required way‑points.  A few sanity checks are included to
        # allow the script to fail gracefully if an expected object is missing.
        required_keys = [
            'bottom_side_pos', 'bottom_anchor_pos',      # drawer way‑points
            'item3',                                     # assumed “trash”
            'bin'                                        # disposal location
        ]
        for k in required_keys:
            if k not in positions:
                raise KeyError(f"[run] Required key “{k}” not supplied by get_object_positions().")

        bottom_side_pos     = positions['bottom_side_pos']
        bottom_anchor_pos   = positions['bottom_anchor_pos']
        trash_pos           = positions['item3']          # specification says “throw away the trash”
        bin_pos             = positions['bin']

        # ------------------------------------------------------------------
        #  Step‑1  rotate gripper to 90 deg about z‑axis
        # ------------------------------------------------------------------
        #   RLBench’s quaternion convention is (x, y, z, w)
        quat_90_deg = R.from_euler('z', 90, degrees=True).as_quat()
        print("\n[Plan‑1] rotate gripper → 90 deg")
        obs, reward, done = rotate(env, task, target_quat=quat_90_deg)
        if done:
            print("[Task] Episode finished prematurely after rotate."); return

        # ------------------------------------------------------------------
        #  Step‑2  move to side‑handle position of bottom drawer
        # ------------------------------------------------------------------
        print("\n[Plan‑2] move → bottom_side_pos =", bottom_side_pos)
        obs, reward, done = move(env, task, target_pos=bottom_side_pos)
        if done:
            print("[Task] Episode finished prematurely after move‑to‑side."); return

        # ------------------------------------------------------------------
        #  Step‑3  move to anchor‑handle position
        # ------------------------------------------------------------------
        print("\n[Plan‑3] move → bottom_anchor_pos =", bottom_anchor_pos)
        obs, reward, done = move(env, task, target_pos=bottom_anchor_pos)
        if done:
            print("[Task] Episode finished prematurely after move‑to‑anchor."); return

        # ------------------------------------------------------------------
        #  Step‑4  pull the drawer handle (open drawer)
        # ------------------------------------------------------------------
        #   We assume the drawer slides along world‑x (positive).  A 0.12 m pull
        #   usually suffices; adjust if the environment uses another convention.
        print("\n[Plan‑4] pull drawer handle (open)")
        obs, reward, done = pull(env, task, pull_distance=0.12, pull_axis='x')
        if done:
            print("[Task] Episode finished prematurely after pull."); return

        # ------------------------------------------------------------------
        #  Step‑5  move to trash (lying on the table)
        # ------------------------------------------------------------------
        print("\n[Plan‑5] move → trash_pos =", trash_pos)
        obs, reward, done = move(env, task, target_pos=trash_pos)
        if done:
            print("[Task] Episode finished prematurely after move‑to‑trash."); return

        # ------------------------------------------------------------------
        #  Step‑6  pick the trash
        # ------------------------------------------------------------------
        print("\n[Plan‑6] pick trash")
        obs, reward, done = pick(env, task, target_pos=trash_pos)
        if done:
            print("[Task] Episode finished prematurely after pick."); return

        # ------------------------------------------------------------------
        #  Step‑7  move to bin
        # ------------------------------------------------------------------
        print("\n[Plan‑7] move → bin_pos =", bin_pos)
        obs, reward, done = move(env, task, target_pos=bin_pos)
        if done:
            print("[Task] Episode finished prematurely after move‑to‑bin."); return

        # ------------------------------------------------------------------
        #  Step‑8  place trash in the bin
        # ------------------------------------------------------------------
        print("\n[Plan‑8] place trash into bin")
        obs, reward, done = place(env, task, target_pos=bin_pos)
        if done:
            print("[Task] Episode finished successfully!  Reward:", reward)
        else:
            print("[Task] Finished plan but environment did not signal done.")

    except Exception as e:
        print(f"[run] Exception occurred: {e}")
        raise

    finally:
        shutdown_environment(env)
        print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()
