# run_task_drawer_and_disposal.py
#
# Description:
#  - Concrete implementation of the full task:
#       1) Rotate the gripper 90° (required for lateral motion w.r.t. drawer
#          specification).
#       2) Move to the “side” waypoint of the bottom drawer.
#       3) Move to the “anchor” waypoint (directly in front of the drawer
#          handle).
#       4) Pick (grasp) the drawer handle.
#       5) Pull the drawer open.
#       6) Move above an item on the table.
#       7) Pick the item (rubbish).
#       8) Move above the bin.
#       9) Place the rubbish in the bin.
#
#  - The above nine high‑level steps exactly match the specification that was
#    provided (rotate, move, move, pick, pull, move, pick, move, place).
#
#  - Only *existing* skills from `skill_code` are used.  No new low‑level skills
#    are defined.  The code is robust to missing objects and will shut the
#    environment down cleanly in every case.

import math
import traceback
from typing import Dict

import numpy as np
from scipy.spatial.transform import Rotation as R
from pyrep.objects.shape import Shape            # required by skeleton imports
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment
from video import init_video_writers, recording_step, recording_get_observation
from object_positions import get_object_positions

# === Import ALL predefined skills exactly as they exist ===
from skill_code import rotate, move, pick, pull, place


def _quaternion_from_euler(z_deg: float = 0.0,
                           y_deg: float = 0.0,
                           x_deg: float = 0.0) -> np.ndarray:
    """Utility – converts Euler angles in degrees to xyzw quaternion."""
    quat = R.from_euler('zyx',
                        [z_deg, y_deg, x_deg],
                        degrees=True).as_quat()
    return quat.astype(np.float32)


def _safe_lookup(name: str, positions: Dict[str, np.ndarray]):
    """Fetch a position from the dict, raising a helpful error if absent."""
    if name not in positions:
        raise KeyError(f"Required object '{name}' not found in "
                       f"get_object_positions(). Keys: {list(positions)}")
    return positions[name]


def run_task_drawer_and_disposal():
    print("\n==========  STARTING  TASK:  Drawer‑and‑Disposal  ==========")

    # ----------------------------------------------------------
    # Environment setup
    # ----------------------------------------------------------
    env, task = setup_environment()
    try:
        descriptions, obs = task.reset()

        # Optional – video recording
        init_video_writers(obs)
        task.step = recording_step(task.step)
        task.get_observation = recording_get_observation(task.get_observation)

        # ------------------------------------------------------
        # Retrieve all useful object positions up‑front
        # ------------------------------------------------------
        positions = get_object_positions()

        bottom_side_pos   = _safe_lookup('bottom_side_pos',   positions)
        bottom_anchor_pos = _safe_lookup('bottom_anchor_pos', positions)
        item1_pos         = _safe_lookup('item1',             positions)
        bin_pos           = _safe_lookup('bin',               positions)

        # “Approach” helper offsets so that the gripper first arrives slightly
        # above surfaces before descending (positive z == upwards in RLBench).
        ABOVE_OFFSET = np.array([0.0, 0.0, 0.10], dtype=np.float32)

        # ======================================================
        #  Plan Execution (9 steps – matches provided spec)
        # ======================================================

        # Step‑1  ROTATE:  align end‑effector 90° around Z‑axis
        ninety_deg_quat = _quaternion_from_euler(z_deg=90.0)
        print("\n[Step 1] rotate → ninety_deg around Z")
        obs, reward, done = rotate(
            env,
            task,
            target_quat=ninety_deg_quat,
            max_steps=150
        )
        if done:
            print("[Task] Terminated right after rotate!")
            return

        # Step‑2  MOVE:  go to drawer’s “side” waypoint
        print("\n[Step 2] move → bottom_side_pos")
        obs, reward, done = move(
            env,
            task,
            target_pos=bottom_side_pos + ABOVE_OFFSET,
            max_steps=150
        )
        if done:
            print("[Task] Terminated during move‑to‑side!")
            return

        # Step‑3  MOVE:  slide into the “anchor” waypoint (right in front of handle)
        print("\n[Step 3] move → bottom_anchor_pos")
        obs, reward, done = move(
            env,
            task,
            target_pos=bottom_anchor_pos,
            max_steps=150
        )
        if done:
            print("[Task] Terminated during move‑to‑anchor!")
            return

        # Step‑4  PICK:  grasp the drawer handle (anchor position)
        print("\n[Step 4] pick → drawer handle")
        obs, reward, done = pick(
            env,
            task,
            target_pos=bottom_anchor_pos,
            approach_distance=0.05,   # short approach – we’re already close
            max_steps=150,
            approach_axis='z'
        )
        if done:
            print("[Task] Terminated immediately after pick‑handle!")
            return

        # Step‑5  PULL:  pull the drawer straight out along +X (≈20 cm)
        print("\n[Step 5] pull → open drawer")
        obs, reward, done = pull(
            env,
            task,
            pull_distance=0.20,       # 20 cm pull distance
            pull_axis='x',
            max_steps=150
        )
        if done:
            print("[Task] Terminated during pull (drawer)!")
            return

        # ------------------------------------------------------------------
        # Drawer is now open – proceed to dispose of rubbish (one example)
        # ------------------------------------------------------------------

        # Step‑6  MOVE:  hover above item1
        print("\n[Step 6] move → above item1 (rubbish)")
        obs, reward, done = move(
            env,
            task,
            target_pos=item1_pos + ABOVE_OFFSET,
            max_steps=150
        )
        if done:
            print("[Task] Terminated during move‑to‑item!")
            return

        # Step‑7  PICK:  grasp item1
        print("\n[Step 7] pick → item1")
        obs, reward, done = pick(
            env,
            task,
            target_pos=item1_pos,
            approach_distance=0.10,
            max_steps=150,
            approach_axis='z'
        )
        if done:
            print("[Task] Terminated right after pick‑item!")
            return

        # Step‑8  MOVE:  hover above bin
        print("\n[Step 8] move → above bin")
        obs, reward, done = move(
            env,
            task,
            target_pos=bin_pos + ABOVE_OFFSET,
            max_steps=150
        )
        if done:
            print("[Task] Terminated during move‑to‑bin!")
            return

        # Step‑9  PLACE:  drop item into bin
        print("\n[Step 9] place → bin")
        obs, reward, done = place(
            env,
            task,
            target_pos=bin_pos,
            approach_distance=0.10,
            max_steps=150,
            approach_axis='z'
        )

        # === Task finished ===
        if done:
            print("\n[Task] Simulation signalled done – success!")
        else:
            print("\n[Task] Plan executed – environment did not finish (done=False).")

    except Exception as exc:
        print("\n[Task] ERROR during execution:")
        traceback.print_exc()
    finally:
        shutdown_environment(env)
        print("==========  TASK  FINISHED  ==========\n")


if __name__ == "__main__":
    run_task_drawer_and_disposal()
