import numpy as np
from scipy.spatial.transform import Rotation as R

from env import setup_environment, shutdown_environment

# --- low‑level manipulation skills (already provided) ---
from skill_code import rotate, move, pick, pull, place

# --- helper utilities for video and object state recording ---
from video import init_video_writers, recording_step, recording_get_observation

# --- utility for getting object poses from the scene ---
from object_positions import get_object_positions



def _quat_from_z_rotation(deg: float) -> np.ndarray:
    """Utility: return xyzw quaternion describing a pure rotation around Z."""
    return R.from_euler('z', deg, degrees=True).as_quat()


def run_skeleton_task() -> None:
    """
    Executes the oracle plan described in the specification
    1) rotate gripper 0 → 90°
    2) move to drawer side position
    3) move to drawer anchor position
    4) grasp drawer handle
    5) pull drawer open
    6) pick rubbish on table
    7) place rubbish in the bin
    """
    print("===== Starting Combined‑Domain Task =====")
    env, task = setup_environment()

    try:
        # ------------------------------------------------------------------
        # 1)  Environment RESET & helper wrappers
        # ------------------------------------------------------------------
        _, init_obs = task.reset()

        # enable optional video recording
        init_video_writers(init_obs)
        task.step = recording_step(task.step)          # type: ignore
        task.get_observation = recording_get_observation(task.get_observation)  # type: ignore

        # ------------------------------------------------------------------
        # 2)  Acquire object positions once the scene is loaded
        # ------------------------------------------------------------------
        positions = get_object_positions()  # dict: {name : np.ndarray([x,y,z])}

        # mandatory keys – raise explicit error if anything is missing
        required_keys = [
            'waypoint1',                # “nowhere‑pos”
            'bottom_side_pos',          # “side-pos-bottom”
            'bottom_anchor_pos',        # “anchor-pos-bottom”
            'rubbish',
            'bin'
        ]
        missing = [k for k in required_keys if k not in positions]
        if missing:
            raise KeyError(f"Missing object positions for: {missing!r}")

        nowhere_pos        = positions['waypoint1']
        bottom_side_pos    = positions['bottom_side_pos']
        bottom_anchor_pos  = positions['bottom_anchor_pos']
        rubbish_pos        = positions['rubbish']
        bin_pos            = positions['bin']

        # small offsets for safer approach (domain/model dependent)
        DRAWER_PULL_DISTANCE = 0.20     # metres
        PULL_AXIS            = 'y'      # positive Y pulls drawer outward

        # ------------------------------------------------------------------
        # 3)  STEP‑BY‑STEP ORACLE PLAN
        # ------------------------------------------------------------------
        # STEP 1  – rotate gripper from 0° → 90°
        print("\n[PLAN‑1] rotate gripper 0° → 90° about Z")
        target_quat = _quat_from_z_rotation(90.0)      # 90 deg around Z
        obs, reward, done = rotate(env, task, target_quat)
        if done:
            print("[Early‑Exit] done triggered during rotate")
            return

        # STEP 2  – move to side‑position of bottom drawer
        print("\n[PLAN‑2] move gripper to bottom_side_pos")
        obs, reward, done = move(env, task, bottom_side_pos)
        if done:
            print("[Early‑Exit] done triggered during move‑to‑side")
            return

        # STEP 3  – move to anchor‑position (drawer handle)
        print("\n[PLAN‑3] move gripper to bottom_anchor_pos")
        obs, reward, done = move(env, task, bottom_anchor_pos)
        if done:
            print("[Early‑Exit] done triggered during move‑to‑anchor")
            return

        # STEP 4  – grasp the drawer handle (pick‑drawer)
        #          we use the general ‘pick’ primitive on the anchor pose
        print("\n[PLAN‑4] grasp drawer handle at anchor‑pos")
        obs, reward, done = pick(env, task, bottom_anchor_pos,
                                 approach_distance=0.05,
                                 approach_axis='z')   # straight‑down approach
        if done:
            print("[Early‑Exit] done triggered during grasp‑drawer")
            return

        # STEP 5  – pull the drawer outwards
        print("\n[PLAN‑5] pull drawer outward")
        obs, reward, done = pull(env, task,
                                 pull_distance=DRAWER_PULL_DISTANCE,
                                 pull_axis=PULL_AXIS)
        if done:
            print("[Early‑Exit] done triggered during pull")
            return

        # ------------------------------------------------------------------
        #  Drawer is open – now dispose rubbish
        # ------------------------------------------------------------------

        # STEP 6  – pick the rubbish lying on the table
        print("\n[PLAN‑6] pick rubbish from table")
        obs, reward, done = move(env, task, rubbish_pos + np.array([0, 0, 0.10]))  # hover
        if done:
            print("[Early‑Exit] done triggered while hovering over rubbish")
            return

        obs, reward, done = pick(env, task, rubbish_pos,
                                 approach_distance=0.10,
                                 approach_axis='-z')
        if done:
            print("[Early‑Exit] done triggered during rubbish pick")
            return

        # STEP 7  – place rubbish inside the bin
        print("\n[PLAN‑7] place rubbish in bin")
        obs, reward, done = move(env, task, bin_pos + np.array([0, 0, 0.10]))  # hover over bin
        if done:
            print("[Early‑Exit] done triggered while hovering over bin")
            return

        obs, reward, done = place(env, task, bin_pos,
                                  approach_distance=0.10,
                                  approach_axis='-z')
        if done:
            print("[SUCCESS] Task finished – rubbish disposed into bin!")
        else:
            print("[INFO] Plan finished (done False) – check environment success metrics.")

    finally:
        shutdown_environment(env)
        print("===== Shutting down environment =====")


if __name__ == "__main__":
    run_skeleton_task()
