# run_skeleton_task.py  — Filled‑in implementation that follows the oracle plan
#
# NOTE:
#   • Uses ONLY the predefined skills (rotate, move, pull, pick, place)
#   • Executes the seven ordered steps given in the specification
#   • Keeps every original import from the skeleton and adds only what is needed
#   • Gracefully shuts the simulator even if something goes wrong

import numpy as np
from pyrep.objects.shape import Shape                     # kept from skeleton
from pyrep.objects.proximity_sensor import ProximitySensor # kept from skeleton

from env import setup_environment, shutdown_environment

# import the predefined skills (no re‑definitions!)
from skill_code import move, pick, place, rotate, pull

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

# extra utility for quaternion handling used in the rotate step
from scipy.spatial.transform import Rotation as R
import traceback


def _safe_skill_call(skill_fn, *args, **kwargs):
    """Utility that executes a skill and prints a readable traceback on failure."""
    try:
        print(f"[Skill‑Runner] Calling {skill_fn.__name__}...")
        obs, reward, done = skill_fn(*args, **kwargs)
        return obs, reward, done
    except Exception as e:
        print(f"[Skill‑Runner] ERROR inside {skill_fn.__name__}: {e}")
        traceback.print_exc()
        raise


def run_skeleton_task():
    print("===== Starting Skeleton Task =====")

    env, task = setup_environment()

    try:
        # ------------------------------------------------------------------
        # 1) Reset task and wrap for video recording
        # ------------------------------------------------------------------
        descriptions, obs = task.reset()
        init_video_writers(obs)

        # wrap step / get_observation so every call is automatically recorded
        task.step = recording_step(task.step)
        task.get_observation = recording_get_observation(task.get_observation)

        # ------------------------------------------------------------------
        # 2) Obtain all relevant object positions from helper module
        # ------------------------------------------------------------------
        positions = get_object_positions()

        # Defensive checks – make sure the keys we need exist
        required_keys = [
            'bottom_side_pos', 'bottom_anchor_pos',
            'rubbish', 'bin'
        ]
        for k in required_keys:
            if k not in positions:
                raise KeyError(f"[Task‑Setup] Required key '{k}' not present in object_positions!")

        # ------------------------------------------------------------------
        # 3) Execute the oracle plan (7 steps)
        # ------------------------------------------------------------------
        # STEP 1  –  rotate gripper (zero_deg  ➜  ninety_deg)
        #
        # A quaternion that represents a +90° rotation about Z
        # (RLBench gripper convention is usually XYZW)
        quat_90deg_z = R.from_euler('z', 90, degrees=True).as_quat()
        _safe_skill_call(rotate, env, task, target_quat=quat_90deg_z)

        # STEP 2  –  move‑to‑side (nowhere‑pos ➜ side‑pos‑bottom)
        #
        # We simply move the end‑effector to the pre‑recorded “side” waypoint
        side_pos = np.array(positions['bottom_side_pos'])
        _safe_skill_call(move, env, task, target_pos=side_pos)

        # STEP 3  –  move‑to‑anchor (side‑pos‑bottom ➜ anchor‑pos‑bottom)
        anchor_pos = np.array(positions['bottom_anchor_pos'])
        _safe_skill_call(move, env, task, target_pos=anchor_pos)

        # STEP 4  –  pick‑drawer (simulate gripper latching on drawer handle)
        #
        # Using the generic “pick” skill, we approach the anchor position and close the gripper.
        _safe_skill_call(
            pick, env, task,
            target_pos=anchor_pos,
            approach_distance=0.10,      # slightly shorter vertical approach
            approach_axis='z'            # assume we approach from above
        )

        # STEP 5  –  pull drawer open
        #
        # We pull 0.18 m in the +X direction (tuned for typical RLBench drawers).
        _safe_skill_call(
            pull, env, task,
            pull_distance=0.18,
            pull_axis='x'
        )

        # ------------------------------------------------------------------
        # At this point the drawer should be open; proceed to dispose rubbish
        # ------------------------------------------------------------------

        # STEP 6  –  pick rubbish from the table
        rubbish_pos = np.array(positions['rubbish'])
        _safe_skill_call(
            pick, env, task,
            target_pos=rubbish_pos,
            approach_distance=0.12,
            approach_axis='z'
        )

        # STEP 7  –  place rubbish into the bin
        bin_pos = np.array(positions['bin'])
        _safe_skill_call(
            place, env, task,
            target_pos=bin_pos,
            approach_distance=0.12,
            approach_axis='z'
        )

        print("===== Oracle plan executed successfully — Goal achieved! =====")

    finally:
        shutdown_environment(env)
        print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()
