import numpy as np
import traceback
from scipy.spatial.transform import Rotation as R

from env import setup_environment, shutdown_environment
from video import init_video_writers, recording_step, recording_get_observation
from object_positions import get_object_positions

# === Pre-defined skills (DO NOT redefine) ===
from skill_code import move, rotate, pick, pull, place


# --------------------------------------------------------------------------- #
#  Utility helpers
# --------------------------------------------------------------------------- #
def _safe_step(skill_fn, *args, **kwargs):
    """Execute one skill and guarantee simulator shutdown on exception."""
    try:
        return skill_fn(*args, **kwargs)
    except Exception:                              # pylint: disable=broad-except
        print("\n[Error] Exception during skill execution:")
        traceback.print_exc()
        raise


def quat_from_z_rotation(angle_deg: float) -> np.ndarray:
    """Return quaternion (x,y,z,w) for rotation about world-Z by angle_deg."""
    return R.from_euler('z', angle_deg, degrees=True).as_quat()


def _lookup_pos(positions: dict, *candidates):
    """Return the first available position among several candidate keys."""
    for key in candidates:
        if key in positions and positions[key] is not None:
            return positions[key]
    return None


# --------------------------------------------------------------------------- #
#  Main task – executes the oracle plan given in the specification
# --------------------------------------------------------------------------- #
def run_task():
    """
    Oracle plan (required skills only):
        1) rotate gripper zero_deg -> ninety_deg
        2) move  nowhere-pos       -> side-pos-bottom
        3) move  side-pos-bottom   -> anchor-pos-bottom
        4) pick drawer handle (generic pick)
        5) pull drawer open
        6) pick rubbish from table
        7) place rubbish into bin
    """
    print("\n=================  START TASK  =================")
    env, task = setup_environment()

    try:
        # ---------------------------------------------------------- #
        # 0) Reset episode & enable video capture
        # ---------------------------------------------------------- #
        descriptions, obs = task.reset()            # `descriptions` unused
        init_video_writers(obs)

        # Wrap RLBench step / get_observation so that video is recorded
        task.step = recording_step(task.step)
        task.get_observation = recording_get_observation(task.get_observation)

        # ---------------------------------------------------------- #
        # 1) Query all object positions we will need
        # ---------------------------------------------------------- #
        positions = get_object_positions()          # Dict[str, np.ndarray]

        # Drawer approach / grasp points
        bottom_side_pos   = _lookup_pos(
            positions, 'side-pos-bottom', 'bottom_side_pos')
        bottom_anchor_pos = _lookup_pos(
            positions, 'anchor-pos-bottom', 'bottom_anchor_pos')

        # Starting “nowhere” position – fall back to current gripper pose
        nowhere_pos = _lookup_pos(
            positions, 'nowhere-pos', 'nowhere_pos') or obs.gripper_pose[:3]

        # Rubbish object (spec name) or any item*
        rubbish_pos = _lookup_pos(
            positions, 'rubbish', 'item1', 'item2', 'item3')

        # Bin position
        bin_pos = _lookup_pos(positions, 'bin')

        # Sanity checks
        if bottom_side_pos is None or bottom_anchor_pos is None:
            raise RuntimeError("Drawer side / anchor positions are missing "
                               "from object_positions().")
        if rubbish_pos is None:
            raise RuntimeError("No rubbish (or substitute item*) found "
                               "in object_positions().")
        if bin_pos is None:
            raise RuntimeError("Bin position missing in object_positions().")

        # ---------------------------------------------------------- #
        # === ORACLE PLAN ===========================================
        # ---------------------------------------------------------- #

        # Step-1 : Rotate gripper 0° → 90°
        target_quat = quat_from_z_rotation(90.0)
        obs, reward, done = _safe_step(
            rotate, env, task, target_quat=target_quat
        )
        if done:
            print("[Task] Environment ended after Step-1."); return

        # Step-2 : Move to drawer side-approach position
        obs, reward, done = _safe_step(
            move, env, task, target_pos=bottom_side_pos
        )
        if done:
            print("[Task] Environment ended after Step-2."); return

        # Step-3 : Move to drawer anchor (handle) position
        obs, reward, done = _safe_step(
            move, env, task, target_pos=bottom_anchor_pos
        )
        if done:
            print("[Task] Environment ended after Step-3."); return

        # Step-4 : Grasp drawer handle (approach from +Z)
        obs, reward, done = _safe_step(
            pick, env, task,
            target_pos=bottom_anchor_pos,
            approach_axis='z'
        )
        if done:
            print("[Task] Environment ended after Step-4."); return

        # Step-5 : Pull drawer along +X (open ≈ 15 cm)
        obs, reward, done = _safe_step(
            pull, env, task,
            pull_distance=0.15,
            pull_axis='x'
        )
        if done:
            print("[Task] Environment ended after Step-5."); return

        # Optional: release handle (open gripper) – not required by spec but safe
        obs, reward, done = _safe_step(
            place, env, task,
            target_pos=bottom_anchor_pos,
            approach_axis='z'
        )
        if done:
            print("[Task] Environment ended after releasing handle."); return

        # Step-6 : Pick up rubbish from table (approach from +Z)
        obs, reward, done = _safe_step(
            pick, env, task,
            target_pos=rubbish_pos,
            approach_axis='z'
        )
        if done:
            print("[Task] Environment ended after Step-6."); return

        # Step-7 : Place rubbish into bin
        obs, reward, done = _safe_step(
            place, env, task,
            target_pos=bin_pos,
            approach_axis='z'
        )

        # ---------------------------------------------------------- #
        # Finished
        # ---------------------------------------------------------- #
        print("\n=================  TASK COMPLETE  =================")
        if done:
            print(f"[Task] RLBench signalled done=True (reward={reward})")
        else:
            print("[Task] All high-level actions executed (done=False).")

    finally:
        shutdown_environment(env)


# --------------------------------------------------------------------------- #
#  Entry-point
# --------------------------------------------------------------------------- #
if __name__ == "__main__":
    run_task()