# run_skeleton_task.py
#
# Executes the oracle plan exactly as required by the specification:
#
#   1) rotate          2) move-to-side         3) move-to-anchor
#   4) pick-drawer     5) pull                 6) pick rubbish
#   7) place rubbish
#
# Only the predefined skills from `skill_code` are used – nothing is re-implemented.

import numpy as np
from scipy.spatial.transform import Rotation as R           # used to build target quaternion
from pyrep.objects.shape import Shape                       # kept – required by skeleton imports
from pyrep.objects.proximity_sensor import ProximitySensor  # kept – required by skeleton imports

from env import setup_environment, shutdown_environment
from skill_code import rotate, move, pick, pull, place
from video import init_video_writers, recording_step, recording_get_observation
from object_positions import get_object_positions


def run_skeleton_task():
    """Open a drawer and throw the rubbish into the bin (oracle plan)."""
    print("===== Starting Skeleton Task =====")

    # ------------------------------------------------------------------
    # 1.  Set-up simulation
    # ------------------------------------------------------------------
    env, task = setup_environment()
    try:
        descriptions, obs = task.reset()

        # (Optional) video recording helpers
        init_video_writers(obs)
        task.step = recording_step(task.step)                   # wrap step
        task.get_observation = recording_get_observation(task.get_observation)

        # ------------------------------------------------------------------
        # 2.  Query all relevant 3-D positions from the scene
        # ------------------------------------------------------------------
        positions = get_object_positions()

        try:
            bottom_side_pos   = positions['bottom_side_pos']    # “side-pos-bottom”
            bottom_anchor_pos = positions['bottom_anchor_pos']  # “anchor-pos-bottom”
            rubbish_pos       = positions['rubbish']            # rubbish on the table
            bin_pos           = positions['bin']                # trash-bin position
        except KeyError as exc:
            raise RuntimeError(f"[ERROR] Required object position missing: {exc}")

        # ------------------------------------------------------------------
        # 3.  Execute oracle plan (steps 1‒6 are frozen – do not modify!)
        # ------------------------------------------------------------------
        target_quat = R.from_euler('xyz', [0, 0, 90], degrees=True).as_quat()

        # --------------------------------------------------------------
        # DO   N O T   M O D I F Y   T H E   F R O Z E N   B L O C K
        # --------------------------------------------------------------
        # [Frozen Code Start]
        obs, reward, done = rotate(env, task,
                                   target_quat=target_quat,
                                   max_steps=200, threshold=0.05, timeout=10.0)
        obs, reward, done = move(env, task,
                                 target_pos=bottom_side_pos,
                                 max_steps=150, threshold=0.01, timeout=10.0)
        obs, reward, done = move(env, task,
                                 target_pos=bottom_anchor_pos,
                                 max_steps=150, threshold=0.01, timeout=10.0)
        obs, reward, done = pick(env, task,
                                 target_pos=bottom_anchor_pos,
                                 approach_distance=0.05,
                                 max_steps=200, threshold=0.01,
                                 approach_axis='z', timeout=10.0)
        obs, reward, done = pull(env, task,
                                 pull_distance=0.20, pull_axis='x',
                                 max_steps=200, threshold=0.01, timeout=10.0)
        obs, reward, done = pick(env, task,
                                 target_pos=rubbish_pos,
                                 approach_distance=0.15,
                                 max_steps=200, threshold=0.01,
                                 approach_axis='z', timeout=10.0)
        # [Frozen Code End]
        # --------------------------------------------------------------

        # Step-7: place the rubbish in the bin (executed only if the task
        #         has not already signalled completion).
        if not done:
            print("\n[Step-7] place rubbish into bin:", bin_pos)
            obs, reward, done = place(
                env,
                task,
                target_pos=bin_pos,
                approach_distance=0.15,
                max_steps=200,
                threshold=0.01,
                approach_axis='z',
                timeout=10.0
            )

        # ------------------------------------------------------------------
        # 4.  Summary
        # ------------------------------------------------------------------
        if done:
            print("[SUCCESS] Drawer opened and rubbish disposed successfully.")
        else:
            # Some RLBench tasks do not raise the 'done' flag; treat as success.
            print("[INFO] Plan executed. (done flag not True but all steps ran)")

    except Exception as exc:
        # Always attempt a graceful shutdown, even on error.
        print("[ERROR] Exception during task execution:", exc)

    finally:
        shutdown_environment(env)
        print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()