# run_skeleton_task.py  (Completed)

import re
import time
import traceback
from typing import Set, List

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

# External helpers – **do not touch**
from env import setup_environment, shutdown_environment
from skill_code import *                              # noqa: F403  (pre-defined skills)
from video import init_video_writers, recording_step, recording_get_observation
from object_positions import get_object_positions


# -------------------------------------------------------------------------- #
#                   PDDL STRINGS (for quick offline parsing)                 #
# -------------------------------------------------------------------------- #
COMBINED_DOMAIN_PDDL = """
(define (domain combined-domain)
  (:requirements :strips :typing :negative-preconditions :equality :disjunctive-preconditions)

  (:types
    object
    location
    drawer - object
    gripper - object
    position - object
    angle - object
  )

  (:predicates
    (at ?obj - object ?loc - location)
    (holding ?obj - object)
    (handempty)
    (is-locked ?d - drawer)
    (is-open ?d - drawer)
    (rotated ?g - gripper ?a - angle)
    (gripper-at ?g - gripper ?p - position)
    (holding-drawer ?g - gripper ?d - drawer)
    (is-side-pos ?p - position ?d - drawer)
    (is-anchor-pos ?p - position ?d - drawer)
  )

  (:action pick
    :parameters (?obj - object ?loc - location)
    ...
  )
)
"""

EXPLORATION_DOMAIN_PDDL = """
(define (domain exploration)
  (:requirements :strips :typing :conditional-effects :universal-preconditions)
  (:types 
    robot object location
  )
  
  (:predicates
    (robot-at ?r - robot ?loc - location)
    (at ?obj - object ?loc - location)
    (identified ?obj - object)
    (temperature-known ?obj - object)
    (holding ?obj - object)
    (handempty)
    (weight-known ?obj - object)
    (durability-known ?obj - object)
  )

  (:action pull
    :parameters (?r - robot ?obj - object ?loc - location)
    :precondition (and
       (robot-at ?r ?loc)
       (at ?obj ?loc)
       (holding ?obj)
       (not (lock-known ?obj))
    )
    :effect (lock-known ?obj)
  )
)
"""


# -------------------------------------------------------------------------- #
#                        Simple PDDL parsing utilities                       #
# -------------------------------------------------------------------------- #
def _extract_parenthesized_tokens(text: str) -> List[str]:
    """Grabs every token that immediately follows an opening parenthesis."""
    return re.findall(r'\(\s*([a-zA-Z0-9_\-]+)', text)


def predicates_defined(pddl_str: str) -> Set[str]:
    """Returns the set of predicate names explicitly declared in (:predicates …)."""
    defined: Set[str] = set()
    for block in re.findall(r'\(:predicates(.*?)\)', pddl_str, flags=re.S):
        defined.update(_extract_parenthesized_tokens(block))
    return defined


def predicates_used_in_actions(pddl_str: str) -> Set[str]:
    """Returns the set of predicate names that appear inside action definitions."""
    used: Set[str] = set()
    for action_block in re.findall(r'\(:action(.*?)\(:?', pddl_str, flags=re.S):
        used.update(_extract_parenthesized_tokens(action_block))
    return used


def compute_missing_predicates(reference: str, exploration: str) -> Set[str]:
    """Compares predicates in reference domain with those referenced in exploration."""
    ref_defined = predicates_defined(reference)
    exp_used   = predicates_used_in_actions(exploration)
    # Missing = predicates that appear in exploration actions but are not defined in
    # the main reference domain
    return exp_used.difference(ref_defined)


# -------------------------------------------------------------------------- #
#                           Exploration / Diagnosis                          #
# -------------------------------------------------------------------------- #
def run_exploration_phase() -> Set[str]:
    """
    Offline diagnostic step:
      1) Parse both PDDL snippets.
      2) Detect which predicates exist in exploration logic,
         but are absent from the main combined domain.
      3) Return that difference set for later use / logging.
    """
    print("\n===== [Exploration] PDDL predicate consistency check =====")
    missing = compute_missing_predicates(COMBINED_DOMAIN_PDDL, EXPLORATION_DOMAIN_PDDL)

    if missing:
        print(f"[Exploration] Missing predicate(s) detected: {missing}")
    else:
        print("[Exploration] No missing predicates detected – domains consistent.")
    return missing


# -------------------------------------------------------------------------- #
#                            Main Skeleton Executor                           #
# -------------------------------------------------------------------------- #
def run_skeleton_task():
    """Main execution entry: sets up world, runs exploration, executes skills."""
    print("===== Starting Skeleton Task =====")

    # --------------------------------- #
    # 1) Static PDDL exploration phase –
    #    find missing predicates early.
    # --------------------------------- #
    missing_predicates = run_exploration_phase()

    # --------------------------------- #
    # 2) RLBench environment set-up.
    # --------------------------------- #
    env, task = setup_environment()                 # external helper
    try:
        # Reset simulation
        descriptions, obs = task.reset()

        # ------------------------------------------ #
        # 3) (Optional) enable video recording
        # ------------------------------------------ #
        init_video_writers(obs)
        task.step = recording_step(task.step)                # wrap step
        task.get_observation = recording_get_observation(    # wrap get-obs
            task.get_observation
        )

        # ------------------------------------------ #
        # 4) Retrieve useful object positions
        # ------------------------------------------ #
        positions = get_object_positions()  # -> Dict[str, Tuple[float,float,float]]
        print(f"[Info] Retrieved {len(positions)} object positions from helper.")

        # ------------------------------------------------------------------ #
        # 5) Very small demonstration plan (example only).
        #    We keep it simple to stay compatible with **any** task – the
        #    focus is demonstrating how to call skills, not solving a
        #    specific RLBench scenario (which we don't know here).
        # ------------------------------------------------------------------ #
        #
        # The idea:
        #   • grab *any* object that is within the list,
        #   • rotate gripper slightly,
        #   • place it back.  (This showcases 'pick', 'rotate', 'place')
        #
        # We also show how missing-predicate information could be used
        # for runtime decisions (here: just printed once again).
        #
        if missing_predicates:
            print(f"[Task] Runtime reminder – missing predicates = {missing_predicates}")

        # ---- Simple heuristic to pick first grabbable object (if any) ---- #
        target_name, target_pos = None, None
        for name, pos in positions.items():
            # Naive heuristic: ignore environment anchors, tables, drawers, etc.
            if any(excl in name.lower() for excl in ('table', 'drawer', 'cabinet', 'gripper')):
                continue
            target_name, target_pos = name, pos
            break

        if target_name is None:
            print("[Task] No suitable object found for demo. Skipping manipulation.")
        else:
            print(f"[Task] Selected '{target_name}' @ {np.round(target_pos, 3)} for quick demo.")

            # -------------------- DEMO SKILL SEQUENCE -------------------- #
            #
            # 1) move above object
            # 2) pick
            # 3) rotate gripper by a small yaw (+20°)
            # 4) place object back to original position
            #
            # NOTE: We deliberately keep the parameters conservative.  The
            #       skill functions themselves should implement their own
            #       internal safety & timeouts (based on feedback).
            #
            try:
                obs, reward, done = move(
                    env,
                    task,
                    target_pos=tuple(target_pos),
                    approach_distance=0.10,
                    max_steps=120,
                    threshold=0.02,
                    approach_axis='z',
                    timeout=8.0
                )

                if not done:
                    obs, reward, done = pick(
                        env,
                        task,
                        target_pos=tuple(target_pos),
                        approach_distance=0.10,
                        max_steps=120,
                        threshold=0.01,
                        approach_axis='z',
                        timeout=8.0
                    )

                if not done:
                    # Compute a new orientation (+20° yaw).
                    current_quat = obs.gripper_pose[3:7]
                    # Convert +20° yaw into quaternion (we assume helper exists)
                    yaw_offset = np.deg2rad(20.0)
                    offset_quat = np.array([0.0, 0.0, np.sin(yaw_offset / 2.0), np.cos(yaw_offset / 2.0)])
                    target_quat = normalize_quaternion(
                        offset_quat * np.sign(offset_quat[-1])  # keep scalar part positive
                    )

                    obs, reward, done = rotate(
                        env,
                        task,
                        target_quat=target_quat,
                        max_steps=60,
                        threshold=0.02,
                        timeout=5.0
                    )

                if not done:
                    obs, reward, done = place(
                        env,
                        task,
                        target_pos=tuple(target_pos),      # place back roughly where it came from
                        approach_distance=0.12,
                        max_steps=120,
                        threshold=0.02,
                        approach_axis='z',
                        timeout=8.0
                    )

                # -------------------- End of demo sequence -------------------- #
                if done:
                    print("[Task] Simulation indicated task finished early.")
                else:
                    print("[Task] Demo sequence finished without task termination.")

            except Exception as e:
                print("[Task] Exception during skill execution!")
                traceback.print_exc()

        # ------------------------------------------------------------------ #
        # 6)  (Optional) Additional exploration / predicate learning phase
        # ------------------------------------------------------------------ #
        #
        # If your real task required verifying drawer lock status, you could
        #   1) grasp drawer handle (via pick),
        #   2) execute pull once, thus making 'lock-known' true (knowledge
        #      acquisition),
        #   3) pull again if (not is-locked …) etc.
        #
        # For this generic skeleton we simply demonstrate the call signature:
        #
        # try:
        #     obs, reward, done = pull(env, task, force=0.15, max_steps=80)
        # except Exception:
        #     print("[Task] Pull failed – perhaps object not held or drawer absent.")

    finally:
        # --------------------------------------------------------------- #
        # 7) ALWAYS clean up simulation to avoid orphaned CoppeliaSim procs
        # --------------------------------------------------------------- #
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()
