# run_skeleton_task.py (Completed)

import time
import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

# Import every predefined skill exactly as provided
from skill_code import *
from video import init_video_writers, recording_step, recording_get_observation

# Helper that returns a dictionary with every object name → position in the scene
from object_positions import get_object_positions


# ----------------------------------------------------------
# Quaternion Utility (local helper, NOT an additional skill)
# ----------------------------------------------------------
def quaternion_from_euler(roll: float, pitch: float, yaw: float) -> np.ndarray:
    """
    Convert Euler angles (roll, pitch, yaw) [rad] to a quaternion [x, y, z, w].
    We keep it here because rotate(…) takes a quaternion target orientation.
    """
    cy, sy = np.cos(yaw * 0.5), np.sin(yaw * 0.5)
    cp, sp = np.cos(pitch * 0.5), np.sin(pitch * 0.5)
    cr, sr = np.cos(roll * 0.5), np.sin(roll * 0.5)

    qw = cr * cp * cy + sr * sp * sy
    qx = sr * cp * cy - cr * sp * sy
    qy = cr * sp * cy + sr * cp * sy
    qz = cr * cp * sy - sr * sp * cy
    return np.asarray([qx, qy, qz, qw], dtype=np.float32)


def run_skeleton_task() -> None:
    """
    Generic entry-point for the RLBench skeleton task.
    The logic below demonstrates a minimal exploration phase that
    guarantees a valid ‘rotated’ predicate (feedback told us this was
    missing) by explicitly invoking the predefined `rotate` skill.
    """
    print("===== Starting Skeleton Task =====")

    # === 1) Environment Setup ===
    env, task = setup_environment()

    try:
        # Reset task to an initial state
        descriptions, obs = task.reset()

        # (Optional) enable video capture so that every step is recorded
        init_video_writers(obs)

        # Wrap step / get_observation with recording utilities
        original_step, original_get_obs = task.step, task.get_observation
        task.step = recording_step(original_step)
        task.get_observation = recording_get_observation(original_get_obs)

        # === 2) Retrieve object positions for later use (if needed) ===
        positions = get_object_positions()

        # --------------------------------------------------------------
        # 3) EXPLORATION PHASE
        # --------------------------------------------------------------
        # Feedback indicates the predicate “rotated” was missing in the
        # planning state.  We therefore make sure to physically rotate
        # the robot’s gripper so that (rotated gripper ninety_deg)
        # becomes true in the environment.
        #
        # Our rotate(…) skill receives: (env, task, target_quat, …)
        # We build the target quaternion corresponding to a 90° rotation
        # around the world Z-axis (roll = 0, pitch = 0, yaw = π/2).
        #
        # NOTE:
        # • Because this is exploration, we do not depend on knowing
        #   whether the rotation is already satisfied; we simply issue
        #   the command once.
        # • Any additional exploratory actions could be added here in the
        #   same spirit (e.g., move around, pick, etc.).  Keeping it
        #   minimal ensures we respect the instruction to avoid
        #   user-defined skills.
        # --------------------------------------------------------------
        print("\n[Exploration] Rotating the gripper to 90° about the Z axis...")
        ninety_deg_quat = quaternion_from_euler(0.0, 0.0, np.pi / 2.0)

        try:
            obs, reward, done = rotate(env, task, target_quat=ninety_deg_quat)
            if done:
                print("[Exploration] Task reported ‘done’ during rotate.  Stopping early.")
                return
        except Exception as e:
            # If rotation fails, gracefully report and continue: some tasks
            # may allow progress even without it.
            print(f"[Exploration] Warning: rotate() raised an exception → {e}")

        # --------------------------------------------------------------
        # 4) TASK-SPECIFIC PLAN
        # --------------------------------------------------------------
        # You would normally insert the remainder of the oracle plan
        # here, expressed solely through the predefined skills:
        #
        #   • move(...)
        #   • pick(...)
        #   • place(...)
        #   • pull(...)
        #
        # Because we do not have a concrete mission specification in this
        # template, we leave the rest as a no-op.  The primary objective
        # in this iteration was to ensure the ‘rotated’ predicate exists.
        # --------------------------------------------------------------
        print("\n[Plan] No further high-level plan steps defined in this template.")

        # A short sleep to visualise final state if watching the GUI.
        time.sleep(1.0)

    finally:
        # Ensure clean shutdown no matter what
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()