# run_skeleton_task.py (Completed Version – includes a minimal “exploration” phase)

import time
import numpy as np
from pyrep.objects.shape import Shape           # kept for completeness (may be unused)
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

# Import every predefined skill exactly as delivered in skill_code
# (move, pick, place, rotate, pull …)
from skill_code import *

from video import (
    init_video_writers,
    recording_step,
    recording_get_observation,
)

# Utility that returns a dictionary of relevant object names → pose / pos tuples
from object_positions import get_object_positions


def run_skeleton_task() -> None:
    """
    A simple controller that
      1) boots the environment,
      2) performs a short “exploration” phase meant to discover / fix the
         missing predicate reported via planner feedback
         – (rotated gripper zero_deg) –
      3) demonstrates usage of one of the available skills (rotate),
      4) shuts everything down cleanly.

    NOTE:
      •  We purposely keep the logic extremely generic.  In real use-cases
         additional pick / place / pull calls would follow once the drawer is
         known to be in the proper orientation.
      •  This file NEVER defines new low-level skills – it only invokes skills
         coming from `skill_code`.
    """

    print("==========  STARTING SKELETON TASK  ==========")

    # --------------------------------------------------
    # 1)  Environment initialisation
    # --------------------------------------------------
    env, task = setup_environment()
    try:
        # RLBench reset – returns textual descriptions + first observation
        descriptions, obs = task.reset()

        # Optional video recording
        init_video_writers(obs)
        # Wrap step() / get_observation() so that every call is automatically
        # piped to the recorder
        task.step = recording_step(task.step)
        task.get_observation = recording_get_observation(task.get_observation)

        # Retrieve any known object positions (if provided by helper module)
        positions = get_object_positions()
        print("[Info] Known object positions:", positions.keys())

        # --------------------------------------------------
        # 2)  EXPLORATION  – “Which predicate is missing?”
        # --------------------------------------------------
        #
        # Planning feedback told us the planner failed because the INITIAL
        # state did not contain:
        #        (rotated gripper zero_deg)
        #
        # The following block therefore:
        #   • Reads the current gripper quaternion from the first observation
        #   • Converts it to a conventional, *normalised* quaternion
        #   • If that quaternion differs from “identity” ([0,0,0,1]) by more
        #     than a small angle, we call the existing `rotate()` skill so
        #     that—physically—the condition now holds in the simulator.
        #
        # Strictly speaking, the predicate lives in the symbolic layer, not in
        # the simulator.  However, ensuring the robot is actually oriented at
        # zero_deg gives us confidence the predicate *would* be true in the
        # logical world as well.
        #

        # --- helper: quaternion angle-difference to identity -----------------
        def quat_angle_to_identity(q_xyzw: np.ndarray) -> float:
            q_xyzw = q_xyzw / np.linalg.norm(q_xyzw)
            # Identity quaternion: w=1, x=y=z=0 → ⟹ [0,0,0,1]
            dot = np.clip(q_xyzw[3], -1.0, 1.0)  # scalar part
            return 2.0 * np.arccos(dot)

        # Current gripper quaternion (x,y,z,w)
        current_quat = obs.gripper_pose[3:7].copy()
        angle_from_identity = quat_angle_to_identity(current_quat)

        print(f"[Exploration] Current gripper quaternion: {current_quat}")
        print(
            f"[Exploration] Angle to “zero_deg” (identity) ≈ "
            f"{np.degrees(angle_from_identity):.2f}°"
        )

        # Threshold – if already ≦ 5°, assume predicate effectively true
        if angle_from_identity > np.radians(5.0):
            target_quat = np.array([0.0, 0.0, 0.0, 1.0])
            print(
                "[Exploration] The predicate (rotated gripper zero_deg) is "
                "NOT satisfied ‑> invoking rotate() skill …"
            )

            # Using provided rotate() from skill_code
            obs, reward, done = rotate(
                env,
                task,
                target_quat=target_quat,
                max_steps=150,
                threshold=0.03,
                timeout=10.0,
            )

            if done:
                # (Unlikely, but better be safe.)
                print(
                    "[Exploration] Environment signalled 'done' while rotating. "
                    "Terminating early."
                )
                return

            print(
                "[Exploration] Rotation complete – we may now safely assume the "
                "missing predicate is satisfied."
            )
        else:
            print(
                "[Exploration] Predicate already satisfied – no rotation "
                "required."
            )

        # --------------------------------------------------
        # 3)  (Optional)  Domain-specific follow-up actions
        # --------------------------------------------------
        #
        # Place any additional pick / move / pull calls below.  They will
        # depend heavily on the concrete task (e.g., opening a drawer, placing
        # an object in a disposal bin, etc.).  As an illustrative placeholder
        # we merely wait a short moment so the recorder shows the end state.
        #
        print("[Plan] No further high-level actions scripted in this example.")
        time.sleep(1.0)  # purely for visualisation when recording

    finally:
        # --------------------------------------------------
        # 4)  Always shut the environment down gracefully
        # --------------------------------------------------
        shutdown_environment(env)
        print("==========  END OF SKELETON TASK  ==========")


# Execute when run as a script
if __name__ == "__main__":
    run_skeleton_task()