# run_skeleton_task.py  (Completed Version – uses only predefined skills)

import time
import numpy as np
from pyrep.objects.shape import Shape          # kept – may be used by other skills
from pyrep.objects.proximity_sensor import ProximitySensor     # kept – id.

from env import setup_environment, shutdown_environment

# Import every predefined skill exactly once
from skill_code import rotate, move, pick, place, pull

from video import init_video_writers, recording_step, recording_get_observation

# Helper that gives us a dictionary of every relevant object’s pose or extra info
from object_positions import get_object_positions


def run_skeleton_task() -> None:
    """
    Generic runner that (1) boots the environment, (2) explores enough to satisfy
    the “rotated” predicate that was reported missing, and (3) cleanly shuts down.

    Nothing in this routine re-implements or overrides any skill.  We only *call*
    the skills that already exist inside `skill_code`.
    """
    print("===== Starting Skeleton Task =====")

    # ----------------------------------------------------------------------
    # 1)  Environment Setup
    # ----------------------------------------------------------------------
    env, task = setup_environment()        # create simulator + RLBench task
    try:
        # ------------------------------------------------------------------
        # 2)  Reset the task & start video recording (optional)
        # ------------------------------------------------------------------
        descriptions, obs = task.reset()   # RLBench returns language + initial obs
        init_video_writers(obs)            # start video writers (if any)

        # Wrap step and observation for automatic video capturing
        original_step_fn = task.step
        task.step = recording_step(original_step_fn)
        original_get_obs_fn = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs_fn)

        # ------------------------------------------------------------------
        # 3)  Gather object position information (generic utility)
        # ------------------------------------------------------------------
        # This may include drawer handles, buttons, disposal bins, etc.
        # We do *not* rely on any particular key here – we simply print what
        # is available for debugging.  Down-stream logic can query as needed.
        try:
            positions = get_object_positions()
            print("[Info] Known object positions:", positions)
        except Exception as e:
            print("[Warning] Could not obtain object positions – proceeding anyway.")
            positions = {}

        # ------------------------------------------------------------------
        # 4)  =====  Exploration Phase  =====
        # ------------------------------------------------------------------
        # Feedback told us that plans were failing because the predicate
        #     (rotated ?g ninety_deg)
        # was never made true.  The skill ‘rotate’ already exists, so here
        # we perform a 90-degree rotation around the Z-axis once at the very
        # beginning of the episode.  This should establish the predicate and
        # let any high-level planner continue unblocked.
        #
        # The quaternion below represents +90° about the global Z axis
        # in (x, y, z, w) = (0, 0, sin(π/4), cos(π/4)).
        # ------------------------------------------------------------------
        target_quat_ninety_deg = np.array([0.0,
                                           0.0,
                                           np.sin(np.pi / 4.0),
                                           np.cos(np.pi / 4.0)])

        print("\n[Exploration] Attempting to satisfy missing predicate (rotated ?g ninety_deg)")
        obs, reward, done = rotate(env,
                                   task,
                                   target_quat=target_quat_ninety_deg,
                                   max_steps=150,
                                   threshold=0.04,
                                   timeout=15.0)

        if done:
            print("[Exploration] Task flagged DONE during rotation – terminating early.")
            return

        print("[Exploration] Rotation completed.  Continuing to task-specific logic.\n")

        # ------------------------------------------------------------------
        # 5)  =====  Placeholder for Task-Specific High-Level Plan  =====
        # ------------------------------------------------------------------
        # At this point, the robot has the correct orientation predicate
        # satisfied.  You can now insert the real (oracle) plan for the
        # combined ‘disposal + open-drawer’ domain here.  Because that oracle
        # plan is not provided in this challenge description, we simply leave
        # a placeholder.  When integrating with your true planner, just
        # replace the block below with successive calls to move / pick /
        # place / pull / …   (all already imported from `skill_code`).
        #
        # Example skeleton calls (commented):
        #
        #   handle_pos = positions.get('drawer_handle')
        #   if handle_pos is not None:
        #       obs, reward, done = move(env, task, target_pos=handle_pos, ... )
        #       if done:
        #           return
        #       obs, reward, done = pick(env, task, target_pos=handle_pos, ... )
        #       ...
        # ------------------------------------------------------------------
        print("[Info] No further high-level plan was given; robot is now idle.")

        # Small idle loop – keeps the simulation alive for a few steps so that
        # the camera can capture a tidy ending to the episode video.
        idle_action = np.zeros(env.action_shape)
        for _ in range(5):
            obs, reward, done = task.step(idle_action)
            if done:
                break

    finally:
        # ------------------------------------------------------------------
        # 6)  Always, *always* shut everything down gracefully
        # ------------------------------------------------------------------
        shutdown_environment(env)
        print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()