# run_skeleton_task.py (Filled-in Skeleton)

import time
import math
import numpy as np

from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

# Bring every predefined skill (pick, place, move, rotate, pull, …) into scope
from skill_code import *

from video import init_video_writers, recording_step, recording_get_observation
from object_positions import get_object_positions



def _quat_from_euler(roll: float, pitch: float, yaw: float):
    """
    Local util – converts intrinsic XYZ (roll-pitch-yaw) to quaternion [x,y,z,w].
    We do NOT add a new skill here; it is only a math helper.
    """
    cy, sy = math.cos(yaw * 0.5), math.sin(yaw * 0.5)
    cp, sp = math.cos(pitch * 0.5), math.sin(pitch * 0.5)
    cr, sr = math.cos(roll * 0.5), math.sin(roll * 0.5)

    w = cr * cp * cy + sr * sp * sy
    x = sr * cp * cy - cr * sp * sy
    y = cr * sp * cy + sr * cp * sy
    z = cr * cp * sy - sr * sp * cy
    return np.asarray([x, y, z, w], dtype=np.float32)



def run_skeleton_task():
    """
    Generic skeleton for running any task in your simulation.
    We augment it with a tiny ‘exploration’ phase that explicitly
    tries to generate the missing «rotated» predicate (see feedback).
    """
    print("===== Starting Skeleton Task =====")

    # === Environment Setup ===
    env, task = setup_environment()

    try:
        # Reset the task to the initial state and collect the very first obs
        descriptions, obs = task.reset()

        # Optional video capture
        init_video_writers(obs)
        task.step = recording_step(task.step)                # Wrap step
        task.get_observation = recording_get_observation(    # Wrap obs
            task.get_observation
        )

        # === Retrieve Object Positions (if needed later) ===
        positions = get_object_positions()
        print("[Info] Known object positions from helper:", positions)

        # ------------------------------------------------------------------
        # EXPLORATION PHASE  –  looking for the missing predicate ‘rotated’
        # ------------------------------------------------------------------
        #
        # Feedback told us the planner complained about ‘rotated (…)’.
        # In the PDDL domain, many later actions (move-to-side, etc.)
        # require  (rotated ?g ninety_deg).  Therefore we demonstrate an
        # exploration that attempts to rotate the gripper by ~90° around
        # the tool’s forward axis.  Once done, every effect-conditioned
        # planning step that demanded ‘rotated’ will be satisfied.
        #
        # NOTE: we do not check symbolic predicates directly (RLBench has
        # no PDDL introspection); we merely execute the physical skill so
        # that the predicate would logically become true.
        # ------------------------------------------------------------------

        print("\n[Exploration] Attempting to satisfy missing predicate ‘rotated’")
        # Choose a quaternion that is 90° (π/2 rad) around Z axis
        target_quat = _quat_from_euler(0.0, 0.0, math.pi * 0.5)
        try:
            obs, reward, done = rotate(
                env,
                task,
                target_quat=target_quat,
                max_steps=150,
                threshold=0.03,
                timeout=15.0,
            )
            if done:
                print("[Exploration] Task finished during rotate – nothing more to do.")
                return
        except Exception as e:
            print(f"[Exploration] Rotate skill threw an exception: {e}")

        # ------------------------------------------------------------------
        # MAIN PLAN (placeholder)
        # ------------------------------------------------------------------
        #
        # Insert subsequent task logic here (pick, place, move, pull …)
        # that relies on the ‘rotated’ predicate being true.
        #
        # Because the concrete benchmark task is unknown in this generic
        # setting, we only print a message instead of hard-coding actions.
        # ------------------------------------------------------------------

        print("\n[Task] No further high-level plan coded – "
              "this skeleton solely demonstrates how to bring the "
              "environment into the state containing (rotated …).")

        # small idle loop to show the environment keeps going
        idle_steps = 10
        zero_action = np.zeros(env.action_shape)
        for i in range(idle_steps):
            obs, reward, done = task.step(zero_action)
            if done:
                print("[Task] Environment signalled done while idling.")
                break
            time.sleep(0.02)

    finally:
        # Always shut down the environment gracefully
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()
