# run_skeleton_task.py (Completed Version – includes a lightweight exploration phase)

import re
import time
import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

# All the predefined low-level skills live here
from skill_code import *         # noqa: F401,F403  (we only *call* skills, never redefine)

# Record / replay helpers (optional, but kept exactly as in the original skeleton)
from video import init_video_writers, recording_step, recording_get_observation

# Utility that gives rough object locations (if available in a particular RLBench task)
from object_positions import get_object_positions


# ---------------------------------------------------------------------------
# 1.  LIGHTWEIGHT PDDL INTROSPECTION – “Which predicates are we missing?”
# ---------------------------------------------------------------------------
COMBINED_DOMAIN_PDDL = r"""
(define (domain combined-domain)
  (:requirements :strips :typing :negative-preconditions :equality :disjunctive-preconditions)
  (:types object location drawer - object gripper - object position - object angle - object)
  (:predicates
    (at ?obj - object ?loc - location)
    (holding ?obj - object)
    (handempty)
    (is-locked ?d - drawer)
    (is-open ?d - drawer)
    (rotated ?g - gripper ?a - angle)
    (gripper-at ?g - gripper ?p - position)
    (holding-drawer ?g - gripper ?d - drawer)
    (is-side-pos ?p - position ?d - drawer)
    (is-anchor-pos ?p - position ?d - drawer)
  )
)
"""

EXPLORATION_DOMAIN_PDDL = r"""
(define (domain exploration)
  (:requirements :strips :typing :conditional-effects :universal-preconditions)
  (:types robot object location)
  (:predicates
    (robot-at ?r - robot ?loc - location)
    (at ?obj - object ?loc - location)
    (identified ?obj - object)
    (temperature-known ?obj - object)
    (holding ?obj - object)
    (handempty)
    (weight-known ?obj - object)
    (durability-known ?obj - object)
  )
)
"""


def _extract_predicates(domain_str):
    """
    Very small helper that extracts predicate names
    from a domain string.  It is intentionally simple and
    therefore robust to minor variations in white-spacing.
    """
    # Grab the text inside the (:predicates … ) block
    block_match = re.search(r"\(:predicates(.*?)\)", domain_str, re.DOTALL)
    if not block_match:
        return set()

    block = block_match.group(1)
    # A predicate line looks like:       (holding ?obj - object)
    # We capture the first token after the opening '('
    names = re.findall(r"\(\s*([^\s\(\)]+)", block)
    return set(names)


def discover_missing_predicates():
    base_preds = _extract_predicates(COMBINED_DOMAIN_PDDL)
    exploration_preds = _extract_predicates(EXPLORATION_DOMAIN_PDDL)
    # Any predicate appearing in exploration, but not in the base, is “missing”
    return sorted(list(exploration_preds - base_preds))


# ---------------------------------------------------------------------------
# 2.  MAIN CONTROL LOOP
# ---------------------------------------------------------------------------
def run_skeleton_task():
    """
    Generic driver:
      • Sets up RLBench environment
      • Runs a tiny exploration routine that inspects PDDL text
        to report *missing* predicates.
      • Executes a very small demo action sequence that only
        relies on predefined skills (rotate / pick / place).
      • Shuts the environment down cleanly.
    """
    print("===== Starting Skeleton Task =====")

    # ------------------------------------------------------
    # Environment bootstrap
    # ------------------------------------------------------
    env, task = setup_environment()          # external helper
    try:
        # ----------------------------------------------
        # Reset RLBench task
        # ----------------------------------------------
        descriptions, obs = task.reset()

        # Initialise (optional) video writers
        init_video_writers(obs)

        # Wrap the task’s step() / get_observation() so that
        # every interaction is automatically recorded
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # --------------------------------------------------
        # 2-A  PDDL Introspection  (exploration phase)
        # --------------------------------------------------
        print("\n----- [Exploration Phase] Looking for missing predicates -----")
        missing_preds = discover_missing_predicates()
        if missing_preds:
            print("  -> Identified predicates that are *absent* from"
                  " the combined domain but appear in the exploration domain:")
            for name in missing_preds:
                print("     •", name)
        else:
            print("  -> No discrepancies detected between the two domains.")
        print("----------------------------------------------------------------\n")

        # --------------------------------------------------
        # 2-B  Obtain some basic world information
        # --------------------------------------------------
        positions = {}
        try:
            positions = get_object_positions() or {}
        except Exception as e:
            print("[Warning] get_object_positions() threw an exception:", e)

        # For demonstration, pick *any* object we can locate and
        # try a rotate + (optional) pick or pull, using predefined skills.
        # If the helper is empty, we simply run a noop-rotate.
        any_obj_name = None
        any_obj_pos = None
        for key, value in positions.items():
            if isinstance(value, (list, tuple, np.ndarray)) and len(value) == 3:
                any_obj_name = key
                any_obj_pos = np.array(value, dtype=float)
                break

        # --------------------------------------------------
        # 2-C  Very small skill demonstration
        # --------------------------------------------------
        # Always demo a quick rotation so that at least one
        # predefined skill is executed.
        print("[Demo] Executing a small rotate() skill call.")
        try:
            # Target == current orientation (so we verify that rotate()
            # recognises we're already aligned and exits quickly)
            target_quat = obs.gripper_pose[3:7]
            rotate(env, task, target_quat, max_steps=20, threshold=0.01)
        except Exception as e:
            print("[Warning] rotate() failed:", e)

        # If we do have an object location, attempt to *pick* and *place*
        # it back down.  We catch *any* runtime error so that the
        # overall script never crashes.
        if any_obj_name is not None:
            print(f"[Demo] Attempting quick pick-and-place on '{any_obj_name}'.")
            try:
                # ----------------------
                # (1) pick()
                # ----------------------
                obs, reward, done = pick(
                    env,
                    task,
                    target_pos=any_obj_pos,
                    approach_distance=0.15,
                    max_steps=100,
                    threshold=0.015,
                    approach_axis='z',
                    timeout=8.0
                )
                if done:
                    print("[Demo] Task ended during pick(); skipping place().")
                else:
                    # ----------------------
                    # (2) place() – drop back where we found it
                    # ----------------------
                    obs, reward, done = place(
                        env,
                        task,
                        target_pos=any_obj_pos,
                        approach_distance=0.15,
                        max_steps=100,
                        threshold=0.02,
                        approach_axis='z',
                        timeout=8.0
                    )
            except Exception as e:
                print(f"[Warning] pick/place attempt failed on '{any_obj_name}':", e)

        # --------------------------------------------------
        # End-of-task clean-up & graceful exit
        # --------------------------------------------------
        print("===== Task Logic Completed Successfully =====")

    finally:
        # Irrespective of what happens, always close simulation
        shutdown_environment(env)
        print("===== RLBench environment shut down =====")


# ---------------------------------------------------------------------------
# 3.  Launch when executed directly
# ---------------------------------------------------------------------------
if __name__ == "__main__":
    run_skeleton_task()
