# run_skeleton_task.py (Completed)

import inspect
import time
import traceback
from typing import Dict, Any

import numpy as np
from scipy.spatial.transform import Rotation as R      # Only used for quaternion helpers

from env import setup_environment, shutdown_environment
from skill_code import *                              # noqa – required by the skeleton
from video import init_video_writers, recording_step, recording_get_observation
from object_positions import get_object_positions


# ------------------------------------------------------------------------------------
# Helper Utilities
# ------------------------------------------------------------------------------------
def euler_to_quat(roll: float, pitch: float, yaw: float):
    """Utility: Convert Euler (XYZ) to quaternion (xyzw)."""
    rot = R.from_euler('xyz', [roll, pitch, yaw])
    q = rot.as_quat()           # scipy returns xyzw by default
    return q


def normalize_quaternion(q):
    """Utility: Normalise a quaternion. Works for xyzw format."""
    q = np.array(q, dtype=np.float64)
    norm = np.linalg.norm(q)
    if norm < 1e-8:
        return q
    return q / norm


def safe_invoke(skill_fn, fallback_return=None, **kwargs):
    """
    A defensive wrapper that calls ANY skill function using only the parameters
    it actually declares (via inspect).  Missing/extra args are ignored so the
    code can work with multiple RLBench templates.
    """
    try:
        sig = inspect.signature(skill_fn)
        accepted_args: Dict[str, Any] = {}
        for name in sig.parameters.keys():
            if name in kwargs:
                accepted_args[name] = kwargs[name]
        return skill_fn(**accepted_args)
    except Exception as e:
        # Print a concise, but informative, traceback instead of hard-crashing.
        print(f"[safe_invoke] WARNING: Exception while running {skill_fn.__name__}: {e}")
        traceback.print_exc()
        return fallback_return


# ------------------------------------------------------------------------------------
# Main Skeleton Logic
# ------------------------------------------------------------------------------------
def run_skeleton_task():
    """
    Generic entry-point which:
      1) boots the RLBench environment,
      2) carries out a minimal exploration routine so that the agent can learn
         predicates such as `lock-known` (see feedback),
      3) demonstrates the use of ONLY the predefined skills (`move, pick,
         place, rotate, pull`),
      4) shuts everything down cleanly.
    """
    print("===== Starting Skeleton Task =====")

    # -----------------------------------------------------
    # Environment setup & video initialisation
    # -----------------------------------------------------
    env, task = setup_environment()
    try:
        descriptions, obs = task.reset()

        # Optional video capture (does nothing if `video/` not required)
        init_video_writers(obs)

        # Wrap task.step & .get_observation so every step is recorded
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # -------------------------------------------------
        # Retrieve a dictionary of object positions
        # Example format: {'drawer_top':  (x,y,z),
        #                  'drawer_handle': (x,y,z), ...}
        # -------------------------------------------------
        positions: Dict[str, Any] = get_object_positions()
        print(f"[INFO] Retrieved {len(positions)} objects from the scene.")

        # Convenient subsets
        drawer_keys = [k for k in positions.keys() if 'drawer' in k]
        movable_keys = [k for k in positions.keys()
                        if k not in drawer_keys and ('object' in k or 'item' in k)]

        # -------------------------------------------------
        # 1)  Exploration Phase – discover predicates
        # -------------------------------------------------
        #
        # We attempt to open drawers.  Whether the drawer is locked or not is
        # unknown *a priori* – precisely the “missing predicate” highlighted by
        # the feedback (`lock-known`).  By interacting (moving, rotating,
        # pulling) we empirically reveal that fact.  The exploration policy is:
        #
        #   For every drawer:
        #       i)   Rotate gripper to 90°
        #       ii)  Move close to drawer’s handle
        #       iii) Pick / grasp the handle
        #       iv)  Pull
        #
        # All steps are executed only with pre-existing skills.  Any failure
        # raises an exception that gets caught by `safe_invoke`, allowing the
        # loop to continue so the robot can keep exploring other drawers.
        #

        print("----- [Phase-1] Drawer exploration / predicate discovery -----")
        ninety_deg_quat = normalize_quaternion(euler_to_quat(0.0, 0.0, np.pi/2.0))

        for handle_key in drawer_keys:
            handle_pos = positions[handle_key]
            print(f"\n[Drawer] Handling '{handle_key}' at pos {np.round(handle_pos, 3)}")

            # 1-a) Rotate gripper to face the handle “side-on” (90° about Z-axis)
            safe_invoke(rotate,
                        env=env,
                        task=task,
                        target_quat=ninety_deg_quat,
                        max_steps=100,
                        threshold=0.05,
                        timeout=10.0)

            # 1-b) Approach (move) the handle
            safe_invoke(move,
                        env=env,
                        task=task,
                        target_pos=handle_pos,
                        approach_distance=0.10,
                        max_steps=120,
                        threshold=0.01,
                        approach_axis='z',
                        timeout=15.0)

            # 1-c) Pick / grasp handle
            safe_invoke(pick,
                        env=env,
                        task=task,
                        target_pos=handle_pos,
                        approach_distance=0.04,
                        max_steps=100,
                        threshold=0.01,
                        approach_axis='z',
                        timeout=10.0)

            # 1-d) Pull to open the drawer (reveals lock state)
            obs, reward, done = safe_invoke(pull,
                                            env=env,
                                            task=task,
                                            pull_distance=0.15,
                                            max_steps=150,
                                            timeout=15.0,
                                            fallback_return=(None, 0.0, False))

            if done:
                print("[Drawer] Task reports completion during pull – aborting exploration.")
                break

        # -------------------------------------------------
        # 2)  Manipulate loose objects (pick-and-place)
        # -------------------------------------------------
        #
        # After drawers are processed (and hopefully unlocked/open), pick up any
        # remaining objects and drop them at a default “goal” position.  The
        # goal is heuristically chosen as the very first location labelled
        # ‘bin’ or, failing that, 20 cm in front of the robot base.
        #
        print("\n----- [Phase-2] Pick-and-place for loose items -----")

        # Determine a generic drop-zone
        drop_zone_key = next((k for k in positions if 'bin' in k), None)
        if drop_zone_key is not None:
            drop_pos = positions[drop_zone_key]
        else:
            # Fallback: use a hard-coded relative coordinate
            drop_pos = np.array([0.5, 0.0, 0.8])
        print(f"[Pick/Place] Using drop-zone at {np.round(drop_pos, 3)}")

        # Iterate over every non-drawer object
        for obj_key in movable_keys:
            obj_pos = positions[obj_key]
            print(f"\n[Object] Processing '{obj_key}' at {np.round(obj_pos, 3)}")

            # Approach & grasp
            safe_invoke(pick,
                        env=env,
                        task=task,
                        target_pos=obj_pos,
                        approach_distance=0.10,
                        max_steps=120,
                        threshold=0.01,
                        approach_axis='z',
                        timeout=10.0)

            # Move above the drop-zone
            safe_invoke(move,
                        env=env,
                        task=task,
                        target_pos=drop_pos,
                        approach_distance=0.10,
                        max_steps=120,
                        threshold=0.01,
                        approach_axis='z',
                        timeout=15.0)

            # Release
            safe_invoke(place,
                        env=env,
                        task=task,
                        target_pos=drop_pos,
                        retreat_distance=0.10,
                        max_steps=100,
                        threshold=0.01,
                        timeout=10.0)

        # -------------------------------------------------
        # 3)  Wrap-up
        # -------------------------------------------------
        print("\n===== Exploration & manipulation finished successfully =====")

    finally:
        shutdown_environment(env)
        print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()
