import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

# Import every predefined skill exactly as provided
from skill_code import move, rotate, pick, place, pull

from video import init_video_writers, recording_step, recording_get_observation
from object_positions import get_object_positions


def run_skeleton_task():
    """Execute the oracle plan that (1) opens a drawer and (2) moves two tomatoes
    from the table onto the plate.

    Required skill‑execution sequence (from the specification):
        1) move   – approach drawer side handle
        2) rotate – rotate gripper 90 deg (side‑grasp orientation)
        3) move   – move to anchor (handle) position
        4) pick   – grasp drawer handle
        5) pull   – slide drawer open
        6) pick   – pick first tomato
        7) place  – place first tomato on plate
        8) pick   – pick second tomato
        9) place  – place second tomato on plate
    """
    print("===== Starting Skeleton Task =====")

    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the environment / task
        descriptions, obs = task.reset()

        # Optional: start video capture
        init_video_writers(obs)
        task.step = recording_step(task.step)
        task.get_observation = recording_get_observation(task.get_observation)

        # === Retrieve Cartesian positions of relevant objects ===
        positions = get_object_positions()
        # Mandatory keys we plan to use
        REQUIRED_KEYS = [
            'bottom_side_pos', 'bottom_anchor_pos',
            'item1', 'item2',           # tomatoes
            'plate'                     # target
        ]
        for key in REQUIRED_KEYS:
            if key not in positions:
                raise KeyError(f"[Task] Missing {key} in object_positions.")

        bottom_side_pos   = np.asarray(positions['bottom_side_pos'], dtype=np.float32)
        bottom_anchor_pos = np.asarray(positions['bottom_anchor_pos'], dtype=np.float32)
        tomato1_pos       = np.asarray(positions['item1'], dtype=np.float32)
        tomato2_pos       = np.asarray(positions['item2'], dtype=np.float32)
        plate_pos         = np.asarray(positions['plate'], dtype=np.float32)

        # Small z‑offset for placing so that we drop the tomato slightly above plate
        plate_place_pos   = plate_pos.copy()
        plate_place_pos[2] += 0.03

        # === Step‑by‑Step Execution of the Oracle Plan ===
        done = False

        # ------------------------------------------------------------------
        # Step 1: move → drawer side position
        # ------------------------------------------------------------------
        print("\n[Plan‑Step 1] move → bottom_side_pos")
        obs, reward, done = move(
            env, task,
            target_pos=bottom_side_pos,
            max_steps=150, threshold=0.01, timeout=10.0
        )
        if done:
            print("[Task] Terminated after step 1"); return

        # ------------------------------------------------------------------
        # Step 2: rotate → rotate gripper 90 deg about Z
        # ------------------------------------------------------------------
        print("\n[Plan‑Step 2] rotate → 90 deg (about Z)")
        target_quat = np.array([0, 0, np.sin(np.pi/4), np.cos(np.pi/4)], dtype=np.float32)
        obs, reward, done = rotate(
            env, task,
            target_quat=target_quat,
            max_steps=100, threshold=0.05, timeout=10.0
        )
        if done:
            print("[Task] Terminated after step 2"); return

        # ------------------------------------------------------------------
        # Step 3: move → drawer anchor (handle) position
        # ------------------------------------------------------------------
        print("\n[Plan‑Step 3] move → bottom_anchor_pos")
        obs, reward, done = move(
            env, task,
            target_pos=bottom_anchor_pos,
            max_steps=150, threshold=0.01, timeout=10.0
        )
        if done:
            print("[Task] Terminated after step 3"); return

        # ------------------------------------------------------------------
        # Step 4: pick → grasp drawer handle
        # ------------------------------------------------------------------
        print("\n[Plan‑Step 4] pick → drawer handle (anchor)")
        obs, reward, done = pick(
            env, task,
            target_pos=bottom_anchor_pos,
            approach_distance=0.12,
            max_steps=150, threshold=0.01,
            approach_axis='z', timeout=10.0
        )
        if done:
            print("[Task] Terminated after step 4"); return

        # ------------------------------------------------------------------
        # Step 5: pull → slide drawer open (along +X, 12 cm)
        # ------------------------------------------------------------------
        print("\n[Plan‑Step 5] pull → open drawer (along +X)")
        obs, reward, done = pull(
            env, task,
            pull_distance=0.12, pull_axis='x',
            max_steps=150, threshold=0.01, timeout=10.0
        )
        if done:
            print("[Task] Terminated after step 5"); return

        # After pull the gripper is still closed; move upward a bit and open
        lift_after_pull = obs.gripper_pose[:3].copy()
        lift_after_pull[2] += 0.08
        obs, reward, done = move(
            env, task,
            target_pos=lift_after_pull,
            max_steps=100, threshold=0.01, timeout=8.0
        )
        if done:
            print("[Task] Terminated after lift post‑pull"); return
        # Opening gripper implicitly occurs in next pick call.

        # ------------------------------------------------------------------
        # Step 6: pick → first tomato
        # ------------------------------------------------------------------
        print("\n[Plan‑Step 6] pick → tomato 1")
        obs, reward, done = pick(
            env, task,
            target_pos=tomato1_pos,
            approach_distance=0.15,
            max_steps=150, threshold=0.01,
            approach_axis='z', timeout=10.0
        )
        if done:
            print("[Task] Terminated after picking tomato 1"); return

        # ------------------------------------------------------------------
        # Step 7: place → tomato 1 onto plate
        # ------------------------------------------------------------------
        print("\n[Plan‑Step 7] place → tomato 1 on plate")
        obs, reward, done = place(
            env, task,
            target_pos=plate_place_pos,
            approach_distance=0.15,
            max_steps=150, threshold=0.01,
            approach_axis='z', timeout=10.0
        )
        if done:
            print("[Task] Terminated after placing tomato 1"); return

        # ------------------------------------------------------------------
        # Step 8: pick → second tomato
        # ------------------------------------------------------------------
        print("\n[Plan‑Step 8] pick → tomato 2")
        obs, reward, done = pick(
            env, task,
            target_pos=tomato2_pos,
            approach_distance=0.15,
            max_steps=150, threshold=0.01,
            approach_axis='z', timeout=10.0
        )
        if done:
            print("[Task] Terminated after picking tomato 2"); return

        # ------------------------------------------------------------------
        # Step 9: place → tomato 2 onto plate
        # ------------------------------------------------------------------
        print("\n[Plan‑Step 9] place → tomato 2 on plate")
        obs, reward, done = place(
            env, task,
            target_pos=plate_place_pos,
            approach_distance=0.15,
            max_steps=150, threshold=0.01,
            approach_axis='z', timeout=10.0
        )
        if done:
            print("[Task] Terminated after placing tomato 2"); return

        print("\n===== Task completed successfully! =====")

    except Exception as e:
        print(f"[Task] Exception encountered: {e}")

    finally:
        # Ensure the environment is cleaned up
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()
