import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

# Import every predefined skill exactly as provided
from skill_code import move, pick, place, rotate, pull

from video import init_video_writers, recording_step, recording_get_observation

# Utility that must already be available
from object_positions import get_object_positions


def _safe_get_pos(name_map, key):
    """Retrieve a position array from the position‑dictionary with error handling."""
    if key not in name_map:
        raise KeyError(f"[run_task] Could not find key '{key}' in object_positions. "
                       f"Available keys: {list(name_map.keys())}")
    return np.asarray(name_map[key], dtype=float)


def _yaw_quaternion(deg):
    """Return a unit quaternion representing a pure yaw‐rotation."""
    from scipy.spatial.transform import Rotation as R
    return R.from_euler('z', deg, degrees=True).as_quat()  # xyzw order


def run_skeleton_task():
    """Concrete task execution following the oracle plan."""
    print("===== Starting Skeleton Task =====")

    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the environment
        descriptions, obs = task.reset()

        # --- Optional video recording initialisation ---
        init_video_writers(obs)
        task.step = recording_step(task.step)
        task.get_observation = recording_get_observation(task.get_observation)

        # === Object positions ===
        positions = get_object_positions()

        # Mapping between PDDL‑style names used in the specification
        # and the identifiers that exist in the positions‑dictionary.
        name_alias = {
            # Drawer helper positions
            "side_pos_bottom":  "bottom_side_pos",
            "anchor_pos_bottom": "bottom_anchor_pos",

            # Tomatoes / Items
            "tomato1": "item1",
            "tomato2": "item2",
            "tomato3": "item3",
            "tomato1_pos": "item1",
            "tomato2_pos": "item2",
            "tomato3_pos": "item3",
        }

        # Convenience lambda that first resolves alias, then fetches coordinates
        alias_pos = lambda key: _safe_get_pos(positions, name_alias.get(key, key))

        # ------------------------------------------------------------------
        # STEP‑BY‑STEP EXECUTION (mirrors the Specification, with minimal
        # extra steps that are practically required for the low‑level skills)
        # ------------------------------------------------------------------

        # ----- Drawer opening sequence -----
        try:
            # Step 1 : move to side_pos_bottom
            print("\n[Plan] Step 1 – move to bottom_side_pos")
            obs, reward, done = move(env, task, alias_pos("side_pos_bottom"))
            if done:
                print("[Task] Terminated while moving to drawer side. Aborting.")
                return

            # Step 2 : rotate gripper to 90 deg (yaw)
            print("\n[Plan] Step 2 – rotate gripper 90°")
            ninety_quat = _yaw_quaternion(90.0)
            obs, reward, done = rotate(env, task, ninety_quat)
            if done:
                print("[Task] Terminated during rotate. Aborting.")
                return

            # Step 3 : move to anchor_pos_bottom (handle)
            print("\n[Plan] Step 3 – move to bottom_anchor_pos (handle)")
            obs, reward, done = move(env, task, alias_pos("anchor_pos_bottom"))
            if done:
                print("[Task] Terminated while approaching handle. Aborting.")
                return

            # (Extra) – close gripper on the handle using pick primitive so that
            # pull starts with a closed gripper.
            print("\n[Plan] Extra – grasping drawer handle (pick)")
            obs, reward, done = pick(env, task, alias_pos("anchor_pos_bottom"))
            if done:
                print("[Task] Terminated while grasping handle. Aborting.")
                return

            # Step 4 : pull the drawer open (pull along +x by 0.15 m)
            print("\n[Plan] Step 4 – pull drawer by 0.15 m")
            obs, reward, done = pull(env, task, pull_distance=0.15, pull_axis='x')
            if done:
                print("[Task] Terminated while pulling drawer. Aborting.")
                return

        except Exception as drawer_err:
            print(f"[Error] Drawer‑handling sequence failed: {drawer_err}")
            return

        # ----- Tomato placement sequence -----
        tomato_keys = ["tomato1", "tomato2", "tomato3"]
        plate_pos = alias_pos("plate")

        for t_key in tomato_keys:
            try:
                print(f"\n[Plan] Handling {t_key}")

                # Current position of the tomato
                tgt_pos = alias_pos(f"{t_key}_pos" if f"{t_key}_pos" in name_alias else name_alias.get(t_key, t_key))

                # 5) move above tomato
                obs, reward, done = move(env, task, tgt_pos)
                if done:
                    print("[Task] Terminated during move to tomato. Aborting.")
                    return

                # 6) pick tomato
                obs, reward, done = pick(env, task, tgt_pos)
                if done:
                    print("[Task] Terminated during pick. Aborting.")
                    return

                # 7) move to plate
                obs, reward, done = move(env, task, plate_pos)
                if done:
                    print("[Task] Terminated during move to plate. Aborting.")
                    return

                # 8) place on plate
                obs, reward, done = place(env, task, plate_pos)
                if done:
                    print("[Task] Terminated during place. Aborting.")
                    return

            except KeyError as k_err:
                print(f"[Warning] Skipping {t_key}: {k_err}")
            except Exception as obj_err:
                print(f"[Error] Failed to manipulate {t_key}: {obj_err}")

        print("\n===== Task Completed Successfully – Drawer opened and tomatoes on plate =====")

    finally:
        # Ensure the simulator shuts down cleanly no matter what
        shutdown_environment(env)


if __name__ == "__main__":
    run_skeleton_task()
