import numpy as np
from scipy.spatial.transform import Rotation as R

# === RLBench / Task Modules ===
from env import setup_environment, shutdown_environment
from video import init_video_writers, recording_step, recording_get_observation
from object_positions import get_object_positions
from skill_code import rotate, move, pull, pick, place   # already available

# ------------------------------ #
#   Helper / Utility Functions   #
# ------------------------------ #
def quat_from_euler(roll=0.0, pitch=0.0, yaw=0.0, degrees=True):
    """Utility – convert euler → quaternion (xyzw order)."""
    return R.from_euler('xyz', [roll, pitch, yaw], degrees=degrees).as_quat()

def lookup(positions, *keys):
    """
    Try multiple symbolic keys in order until one exists in the positions dict.
    This is useful because PDDL‑style names (anchor-pos-bottom) sometimes map to
    Python object names (bottom_anchor_pos) in the RLBench scene.
    """
    for k in keys:
        if k in positions:
            return positions[k]
    raise KeyError(f"None of the candidate keys {keys} found in object‑positions!")

# ------------------------------ #
#           Main Task            #
# ------------------------------ #
def run_skeleton_task():
    """
    Task Logic:
      1. Rotate gripper to 90 deg (around Z) so it can fit into the drawer side.
      2. Move to the ‘side’ position of the bottom drawer, then to the ‘anchor’
         (handle) position.
      3. Pull the drawer open.
      4. Pick two “tomato” items (item1, item2) from the table and place them
         on the plate.
    """
    print("\n==========  START COMBINED TASK  ==========")

    # --------------------------------------------------
    #  Environment setup
    # --------------------------------------------------
    env, task = setup_environment()
    done = False
    try:
        descriptions, obs = task.reset()

        # Optionally start video recording
        init_video_writers(obs)
        task.step = recording_step(task.step)                 # wrap step() for video
        task.get_observation = recording_get_observation(     # wrap get_observation()
            task.get_observation)

        # --------------------------------------------------
        #  Acquire object positions from the RLBench scene
        # --------------------------------------------------
        positions = get_object_positions()

        # Drawer related key resolution
        side_bottom_pos   = lookup(positions, 'side-pos-bottom',   'bottom_side_pos')
        anchor_bottom_pos = lookup(positions, 'anchor-pos-bottom', 'bottom_anchor_pos')

        # Item & plate positions
        item1_pos = lookup(positions, 'tomato1', 'item1')
        item2_pos = lookup(positions, 'tomato2', 'item2')
        plate_pos = lookup(positions, 'plate')

        # --------------------------------------------------
        #  PLAN EXECUTION (follow oracle specification)
        # --------------------------------------------------
        # Step‑1 : rotate to 90 deg around Z (ninety_deg)
        print("\n[Plan] Step 1 – rotate to ninety_deg")
        target_quat = quat_from_euler(0, 0, 90)   # 90° yaw
        obs, reward, done = rotate(env, task, target_quat)
        if done: return

        # Step‑2 : move to side‑pos‑bottom
        print("\n[Plan] Step 2 – move to side‑pos‑bottom")
        obs, reward, done = move(env, task, side_bottom_pos)
        if done: return

        # Step‑3 : move to anchor‑pos‑bottom
        print("\n[Plan] Step 3 – move to anchor‑pos‑bottom")
        obs, reward, done = move(env, task, anchor_bottom_pos)
        if done: return

        # Step‑4 : pull the drawer handle backwards (‑x dir)
        print("\n[Plan] Step 4 – pull drawer open")
        # Heuristic: pull 0.15 m along ‑x (drawer slides outwards along –x)
        obs, reward, done = pull(env, task, pull_distance=0.15, pull_axis='-x')
        if done: return

        # ----------------  Tomato #1 ---------------- #
        # Step‑5 : move above tomato1
        print("\n[Plan] Step 5 – move above item1 (tomato1)")
        # Slightly raise target z so we approach from above
        tomato1_above = np.copy(item1_pos)
        tomato1_above[2] += 0.10        # +10 cm in Z
        obs, reward, done = move(env, task, tomato1_above)
        if done: return

        # Step‑6 : pick tomato1
        print("\n[Plan] Step 6 – pick tomato1")
        obs, reward, done = pick(env, task, item1_pos, approach_axis='-z')
        if done: return

        # Step‑7 : move above the plate
        print("\n[Plan] Step 7 – move above plate")
        plate_above = np.copy(plate_pos)
        plate_above[2] += 0.10
        obs, reward, done = move(env, task, plate_above)
        if done: return

        # Step‑8 : place tomato1 on plate
        print("\n[Plan] Step 8 – place tomato1")
        obs, reward, done = place(env, task, plate_pos, approach_axis='-z')
        if done: return

        # ----------------  Tomato #2 ---------------- #
        # Step‑9 : move above tomato2
        print("\n[Plan] Step 9 – move above item2 (tomato2)")
        tomato2_above = np.copy(item2_pos)
        tomato2_above[2] += 0.10
        obs, reward, done = move(env, task, tomato2_above)
        if done: return

        # Step‑10 : pick tomato2
        print("\n[Plan] Step 10 – pick tomato2")
        obs, reward, done = pick(env, task, item2_pos, approach_axis='-z')
        if done: return

        # Step‑11 : move above plate
        print("\n[Plan] Step 11 – move above plate (again)")
        obs, reward, done = move(env, task, plate_above)
        if done: return

        # Step‑12 : place tomato2
        print("\n[Plan] Step 12 – place tomato2")
        obs, reward, done = place(env, task, plate_pos, approach_axis='-z')
        if done: return

        # --------------------------------------------------
        #  SUCCESS
        # --------------------------------------------------
        print("\n==========  TASK COMPLETED SUCCESSFULLY!  ==========")

    except Exception as e:
        print(f"[ERROR] Exception encountered: {e}")

    finally:
        shutdown_environment(env)
        print("==========  SHUTDOWN COMPLETE  ==========\n")

# Execute when file is run standalone
if __name__ == "__main__":
    run_skeleton_task()
