# run_combined_task.py
#
# This script executes the oracle plan described in the specification
# by sequentially calling the predefined skills contained in
# skill_code (rotate, move, pick, pull, place).
#
# The high‑level plan is:
#   1) Rotate the gripper to 90 deg.
#   2) Move to the side position of the middle drawer.
#   3) Move to the anchor position of the middle drawer.
#   4) Grasp (pick) the drawer handle.
#   5) Pull the drawer open.
#   6) Pick up tomato 1 from the table and place it on the plate.
#   7) Pick up tomato 2 from the table and place it on the plate.
#
# NOTE:
#   • No new skills are defined – only the existing ones are used.
#   • All object poses are obtained via object_positions.get_object_positions().
#   • Any failure raises an exception which will automatically shut
#     down the environment in the finally‑block.

import numpy as np
from scipy.spatial.transform import Rotation as R
from pyrep.objects.shape import Shape      # kept for completeness (may be unused)
from pyrep.objects.proximity_sensor import ProximitySensor  # kept for completeness

from env import setup_environment, shutdown_environment
from skill_code import *                  # rotate, move, pick, pull, place …
from video import (init_video_writers,
                   recording_step,
                   recording_get_observation)
from object_positions import get_object_positions


# -----------------------------------------------------------
# Helper utilities
# -----------------------------------------------------------

def _fetch_position(positions, key):
    """Return a copy of the requested object position as a numpy array.

    Args:
        positions (dict): name → np.ndarray([x, y, z])
        key (str)       : object name to look up

    Raises:
        RuntimeError: if the key is missing
    """
    if key not in positions:
        raise RuntimeError(f"[Planner] Cannot find position for object “{key}”. "
                           f"Available keys: {list(positions.keys())}")
    return np.array(positions[key], dtype=np.float32).copy()


def _current_quat(task):
    """Return the current gripper quaternion (xyzw) as a numpy array."""
    return np.asarray(task.get_observation().gripper_pose[3:7], dtype=np.float32)


# -----------------------------------------------------------
# Main execution logic
# -----------------------------------------------------------

def run_combined_task():
    print("\n=====  Combined Task: Drawer + Tomato Disposal  =====")

    # ---------------- Environment initialisation ----------------
    env, task = setup_environment()
    try:
        descriptions, obs = task.reset()

        # ‑‑ Optional video recorder
        init_video_writers(obs)
        task.step = recording_step(task.step)
        task.get_observation = recording_get_observation(task.get_observation)

        # Get all known object positions as a dictionary
        positions = get_object_positions()

        # -----------   Resolve positions required by the plan -----------
        side_pos_middle    = _fetch_position(positions, 'middle_side_pos')
        anchor_pos_middle  = _fetch_position(positions, 'middle_anchor_pos')
        tomato1_pos        = _fetch_position(positions, 'tomato1')
        tomato2_pos        = _fetch_position(positions, 'tomato2')
        plate_pos          = _fetch_position(positions, 'plate')

        # --------------------  Execute the oracle plan ------------------

        # Step 1 – rotate gripper from zero_deg → ninety_deg
        print("\n[Step 1] rotate gripper → 90°")
        start_quat = _current_quat(task)
        # rotate 90 deg about world‑Z axis
        target_quat = (R.from_quat(start_quat) *
                       R.from_euler('z', 90.0, degrees=True)).as_quat()
        obs, reward, done = rotate(env, task, target_quat)
        if done:
            print("[Task] Finished unexpectedly after rotate.")
            return

        # Step 2 – move to side‑pos‑middle
        print("\n[Step 2] move → middle_side_pos")
        obs, reward, done = move(env, task, side_pos_middle)
        if done:
            print("[Task] Finished unexpectedly after move 2.")
            return

        # Step 3 – move to anchor‑pos‑middle
        print("\n[Step 3] move → middle_anchor_pos")
        obs, reward, done = move(env, task, anchor_pos_middle)
        if done:
            print("[Task] Finished unexpectedly after move 3.")
            return

        # Step 4 – pick (grasp drawer handle)
        print("\n[Step 4] pick drawer handle at anchor position")
        obs, reward, done = pick(env, task,
                                 target_pos=anchor_pos_middle,
                                 approach_distance=0.10,
                                 approach_axis='z')
        if done:
            print("[Task] Finished unexpectedly after pick drawer.")
            return

        # Step 5 – pull drawer open
        print("\n[Step 5] pull drawer along +x (≈ 0.12 m)")
        obs, reward, done = pull(env, task,
                                 pull_distance=0.12,
                                 pull_axis='x')
        if done:
            print("[Task] Finished unexpectedly after pull.")
            return

        # Step 6 – pick tomato 1
        print("\n[Step 6] pick tomato1")
        obs, reward, done = move(env, task, tomato1_pos + np.array([0, 0, 0.15]))
        if done:
            print("[Task] Finished unexpectedly while repositioning to tomato1.")
            return
        obs, reward, done = pick(env, task,
                                 target_pos=tomato1_pos,
                                 approach_distance=0.12,
                                 approach_axis='-z')
        if done:
            print("[Task] Finished unexpectedly after picking tomato1.")
            return

        # Step 7 – place tomato 1 on plate
        print("\n[Step 7] place tomato1 on plate")
        obs, reward, done = place(env, task,
                                  target_pos=plate_pos,
                                  approach_distance=0.12,
                                  approach_axis='-z')
        if done:
            print("[Task] Finished unexpectedly after placing tomato1.")
            return

        # Step 8 – pick tomato 2
        print("\n[Step 8] pick tomato2")
        obs, reward, done = move(env, task, tomato2_pos + np.array([0, 0, 0.15]))
        if done:
            print("[Task] Finished unexpectedly while repositioning to tomato2.")
            return
        obs, reward, done = pick(env, task,
                                 target_pos=tomato2_pos,
                                 approach_distance=0.12,
                                 approach_axis='-z')
        if done:
            print("[Task] Finished unexpectedly after picking tomato2.")
            return

        # Step 9 – place tomato 2 on plate
        print("\n[Step 9] place tomato2 on plate")
        obs, reward, done = place(env, task,
                                  target_pos=plate_pos,
                                  approach_distance=0.12,
                                  approach_axis='-z')
        if done:
            print("[Task] Task completed (done=True). Reward:", reward)
        else:
            print("[Task] Plan finished – environment not signalling done.")

    finally:
        # Clean shutdown even if an error occurs
        shutdown_environment(env)

    print("=====  Task script finished  =====")


if __name__ == "__main__":
    run_combined_task()