import numpy as np
from scipy.spatial.transform import Rotation as R

from env import setup_environment, shutdown_environment
from skill_code import move, pick, place, pull, rotate
from video import init_video_writers, recording_step, recording_get_observation
from object_positions import get_object_positions


def _lookup_position(name_map, positions_dict):
    """
    Helper that tries a list of possible keys and returns the first hit.
    Raises a KeyError if none are found.
    """
    for k in name_map:
        if k in positions_dict:
            return np.asarray(positions_dict[k], dtype=np.float32)
    raise KeyError(f"None of the keys {name_map} found in object positions.")


def run_skeleton_task():
    """
    Executes the drawer‑opening / tomato‑placing scenario exactly as described in
    the specification.  The sequence of skills is:

      1  rotate gripper from zero_deg → ninety_deg
      2  move  nowhere‑pos  → side‑pos‑bottom
      3  move  side‑pos‑bottom → anchor‑pos‑bottom
      4  pick  the drawer handle (bottom drawer)
      5  pull  the drawer open
      6  pick  tomato1  & place on plate
      7  pick  tomato2  & place on plate
    """
    print("=====   Combined‑Domain Task  =====")

    # ----------------------------------------------------------------------
    # Environment initialisation
    # ----------------------------------------------------------------------
    env, task = setup_environment()
    try:
        descriptions, obs = task.reset()

        # ---------- optional video recording ----------
        init_video_writers(obs)
        task.step = recording_step(task.step)
        task.get_observation = recording_get_observation(task.get_observation)

        # ------------------------------------------------------------------
        # Retrieve all relevant positions from RLBench helpers
        # ------------------------------------------------------------------
        positions = get_object_positions()

        # Drawer related handles
        side_pos_bottom   = _lookup_position(
            ["side-pos-bottom", "bottom_side_pos"], positions
        )
        anchor_pos_bottom = _lookup_position(
            ["anchor-pos-bottom", "bottom_anchor_pos"], positions
        )

        # Tomatoes & plate
        tomato1_pos = _lookup_position(["tomato1"], positions)
        tomato2_pos = _lookup_position(["tomato2"], positions)
        plate_pos   = _lookup_position(["plate"],   positions)

        # Waypoint that is safely out of the workspace (“nowhere”)
        nowhere_pos = _lookup_position(["nowhere-pos", "waypoint1"], positions)

        # ------------------------------------------------------------------
        # 1) ROTATE gripper from zero → 90 deg about the tool‑z axis
        # ------------------------------------------------------------------
        target_quat = R.from_euler("z", 90.0, degrees=True).as_quat()
        rotate(env, task, target_quat)

        # ------------------------------------------------------------------
        # 2) MOVE to the side position of the bottom drawer
        # ------------------------------------------------------------------
        move(env, task, side_pos_bottom)

        # ------------------------------------------------------------------
        # 3) MOVE along the drawer front to the anchor (handle) position
        # ------------------------------------------------------------------
        move(env, task, anchor_pos_bottom)

        # ------------------------------------------------------------------
        # 4) PICK the drawer handle (closes gripper on the anchor)
        # ------------------------------------------------------------------
        pick(env, task, anchor_pos_bottom, approach_axis="-z")

        # ------------------------------------------------------------------
        # 5) PULL the drawer straight out (‑x axis, ~0.15 m)
        # ------------------------------------------------------------------
        pull(env, task, pull_distance=0.15, pull_axis="-x")

        # After opening, release handle to free the gripper
        place(env, task, anchor_pos_bottom, approach_axis="-z")

        # ------------------------------------------------------------------
        # 6) PICK → PLACE tomato1 to plate
        # ------------------------------------------------------------------
        pick(env, task, tomato1_pos, approach_axis="-z")
        place(env, task, plate_pos,   approach_axis="-z")

        # ------------------------------------------------------------------
        # 7) PICK → PLACE tomato2 to plate
        # ------------------------------------------------------------------
        pick(env, task, tomato2_pos, approach_axis="-z")
        place(env, task, plate_pos,  approach_axis="-z")

        print("=====   Task completed successfully!   =====")

    except Exception as e:
        print(f"[ERROR] Exception during task execution: {e}")
        raise  # Re‑raise so upstream can decide how to handle
    finally:
        shutdown_environment(env)


if __name__ == "__main__":
    run_skeleton_task()
