# run_combined_task.py
#
# This file instantiates the RLBench environment, then executes the
# oracle plan given in the specification by invoking ONLY the pre‑defined
# skills that already exist in skill_code.py.
#
# The high–level oracle plan (natural language):
#   1) Rotate the gripper to 90 deg
#   2) Move to the side of the middle drawer
#   3) Move to the anchor of the middle drawer
#   4) Grasp the drawer handle
#   5) Pull the drawer open
#   6) Pick tomato1 – place on plate
#   7) Pick tomato2 – place on plate
#
# The same sequence is reproduced step‑by‑step in this script.

import numpy as np
from scipy.spatial.transform import Rotation as R

from env import setup_environment, shutdown_environment
from video import init_video_writers, recording_step, recording_get_observation
from object_positions import get_object_positions

# === skill imports (do NOT redefine) ===
from skill_code import rotate, move, pick, pull, place


# -------------------------------
# Helper utilities
# -------------------------------
def quat_from_euler_deg(roll=0.0, pitch=0.0, yaw=0.0, seq='xyz'):
    """Return xyzw quaternion given Euler angles in DEGREES."""
    return R.from_euler(seq, [roll, pitch, yaw], degrees=True).as_quat()


def open_gripper(env, task, hold_steps=10):
    """
    Convenience helper: just open the gripper where it is.
    Uses the low‑level action interface directly so that it
    does NOT change the TCP pose.
    """
    obs = task.get_observation()
    action = np.zeros(env.action_shape)
    action[:3] = obs.gripper_pose[:3]
    action[3:7] = obs.gripper_pose[3:7]
    action[-1] = 1.0   # Open

    for _ in range(hold_steps):
        obs, _, done = task.step(action)
        if done:
            break
    return obs


# -------------------------------
# Main task routine
# -------------------------------
def run_combined_task():
    print("\n==========  START  combined‑task  ==========")

    env, task = setup_environment()
    try:
        # Reset & wrap for video
        descriptions, obs = task.reset()
        init_video_writers(obs)

        task.step = recording_step(task.step)
        task.get_observation = recording_get_observation(task.get_observation)

        # ------------------------------------------------------------
        # Acquire all object positions that we will need during the run
        # ------------------------------------------------------------
        pos_dict = get_object_positions()

        # Mapping from PDDL names -> RLBench object keys
        key_map = {
            'nowhere-pos': 'waypoint1',          # starting “parking” pose of the gripper
            'side-pos-middle': 'middle_side_pos',
            'anchor-pos-middle': 'middle_anchor_pos',
            'tomato1': 'tomato1',
            'tomato2': 'tomato2',
            'plate': 'plate'
        }

        # Convenience lambda for looking up a 3‑D target position
        p = lambda name: np.array(pos_dict[key_map[name]], dtype=np.float32)

        # === ORACLE PLAN EXECUTION ===
        #
        # Step‑1  ROTATE gripper zero_deg → ninety_deg
        target_quat = quat_from_euler_deg(yaw=90.0)   # rotate 90 deg about Z
        print("\n--- [1] rotate gripper 0° → 90° ---")
        obs, _, _ = rotate(env, task, target_quat)

        # Step‑2  MOVE gripper nowhere‑pos → side‑pos‑middle
        print("\n--- [2] move to side‑pos‑middle ---")
        obs, _, _ = move(env, task, p('side-pos-middle'))

        # Step‑3  MOVE side‑pos‑middle → anchor‑pos‑middle
        print("\n--- [3] move to anchor‑pos‑middle ---")
        obs, _, _ = move(env, task, p('anchor-pos-middle'))

        # Step‑4  PICK drawer handle at anchor‑pos‑middle
        print("\n--- [4] grasp drawer handle ---")
        obs, _, _ = pick(
            env,
            task,
            target_pos=p('anchor-pos-middle'),
            approach_distance=0.10,
            approach_axis='-y'  # approach along –Y (into the cabinet)
        )

        # Step‑5  PULL drawer open (0.12 m along +X)
        print("\n--- [5] pull drawer open ---")
        obs, _, _ = pull(
            env,
            task,
            pull_distance=0.12,
            pull_axis='x'
        )

        # Optional: release handle so the gripper can pick tomatoes
        print("\n--- release drawer handle ---")
        obs = open_gripper(env, task)

        # Step‑6  PICK tomato1  →  Step‑7 PLACE on plate
        print("\n--- [6] pick tomato1 ---")
        obs, _, _ = pick(
            env,
            task,
            target_pos=p('tomato1'),
            approach_distance=0.10,
            approach_axis='z'
        )

        print("\n--- [7] place tomato1 on plate ---")
        obs, _, _ = place(
            env,
            task,
            target_pos=p('plate'),
            approach_distance=0.10,
            approach_axis='z'
        )

        # Step‑8  PICK tomato2  →  Step‑9 PLACE on plate
        print("\n--- [8] pick tomato2 ---")
        obs, _, _ = pick(
            env,
            task,
            target_pos=p('tomato2'),
            approach_distance=0.10,
            approach_axis='z'
        )

        print("\n--- [9] place tomato2 on plate ---")
        obs, reward, done = place(
            env,
            task,
            target_pos=p('plate'),
            approach_distance=0.10,
            approach_axis='z'
        )

        # Final status
        if done:
            print("\n##### TASK COMPLETED – reward:", reward, "#####")
        else:
            print("\n##### TASK FINISHED but environment reports done=False #####")

    finally:
        shutdown_environment(env)
        print("==========  END  combined‑task  ==========")


if __name__ == "__main__":
    run_combined_task()