# run_combined_task.py
#
# This script instantiates the RL‑Bench environment, then executes the exact
# oracle plan that was provided in the “Specification” section.  All low‑level
# motions are delegated to the already‑implemented skills that come with the
# template (move / rotate / pick / pull / place).  The only responsibility of
# this file is to
#   1) look‑up the Cartesian positions (and, for the first step, the current
#      gripper orientation) that the skills require, and
#   2) chain the skills together in the order given by the oracle plan.
#
# NOTE:
# • No new skills are defined – we only call the ones that already exist.
# • All imports that were present in the skeleton remain unchanged.

import numpy as np
from scipy.spatial.transform import Rotation as R

from env import setup_environment, shutdown_environment
from skill_code import move, pick, place, rotate, pull
from video import init_video_writers, recording_step, recording_get_observation
from object_positions import get_object_positions


# --------------------------------------------------------------------------- #
#                              Helper utilities                               #
# --------------------------------------------------------------------------- #
def quaternion_after_z_rotation(start_quat_xyzw: np.ndarray, angle_deg: float):
    """
    Return a new quaternion that is ‘angle_deg’ degrees rotated about the Z‑axis
    from the orientation given in ‘start_quat_xyzw’.
    """
    start_R = R.from_quat(start_quat_xyzw)
    delta_R = R.from_euler('z', angle_deg, degrees=True)
    target_R = start_R * delta_R
    return target_R.as_quat()


# --------------------------------------------------------------------------- #
#                                 Main logic                                  #
# --------------------------------------------------------------------------- #
def run_combined_task():
    print("\n================  Starting Combined‑Domain Task  ================\n")

    # ---- 1)  Environment initialisation ----------------------------------- #
    env, task = setup_environment()
    try:
        descriptions, obs = task.reset()

        # (Optional) video capture
        init_video_writers(obs)
        task.step = recording_step(task.step)            # wrap step
        task.get_observation = recording_get_observation(
            task.get_observation)                        # wrap observation

        # ---- 2)  Fetch object Cartesian positions ------------------------- #
        #
        # get_object_positions() is assumed to return a dict whose keys match
        # the names in the “Object List” section.
        #
        positions = get_object_positions()

        # Mandatory keys we are going to use – raise a clear error early if
        # they are missing.
        required_keys = [
            'bottom_side_pos', 'bottom_anchor_pos', 'tomato1', 'tomato2',
            'plate'
        ]
        missing = [k for k in required_keys if k not in positions]
        if missing:
            raise RuntimeError(f"[run] Missing keys from get_object_positions(): {missing}")

        # Convenience variables
        side_pos_bottom     = positions['bottom_side_pos']
        anchor_pos_bottom   = positions['bottom_anchor_pos']
        tomato1_pos         = positions['tomato1']
        tomato2_pos         = positions['tomato2']
        plate_pos           = positions['plate']

        # ------------------------------------------------------------------- #
        #                      Oracle plan  –  step‑by‑step                   #
        # ------------------------------------------------------------------- #

        # Step‑1  : rotate gripper from zero‑deg → ninety‑deg
        obs = task.get_observation()
        current_quat = obs.gripper_pose[3:7]          # xyzw
        target_quat  = quaternion_after_z_rotation(current_quat, 90.0)
        print("\n[Plan] Step‑1  rotate gripper  (zero_deg → ninety_deg)")
        obs, reward, done = rotate(env, task, target_quat)
        if done:
            print("[Plan] Task unexpectedly finished after rotate.")
            return

        # Step‑2  : move gripper  nowhere‑pos → side‑pos‑bottom
        print("\n[Plan] Step‑2  move gripper  nowhere‑pos → side‑pos‑bottom")
        obs, reward, done = move(env, task, side_pos_bottom)
        if done:
            print("[Plan] Task unexpectedly finished after move‑to‑side.")
            return

        # Step‑3  : move gripper  side‑pos‑bottom → anchor‑pos‑bottom
        print("\n[Plan] Step‑3  move gripper  side‑pos‑bottom → anchor‑pos‑bottom")
        obs, reward, done = move(env, task, anchor_pos_bottom)
        if done:
            print("[Plan] Task unexpectedly finished after move‑to‑anchor.")
            return

        # Step‑4  : pick drawer handle (close gripper at anchor_pos_bottom)
        #
        #         The generic ‘pick’ skill performs an approach, then closes
        #         the gripper.  That is sufficient for grasping the drawer
        #         handle even though the high‑level PDDL action is named
        #         “pick‑drawer”.
        #
        print("\n[Plan] Step‑4  grasp drawer handle (pick)")
        obs, reward, done = pick(env, task, anchor_pos_bottom, approach_axis='y')
        if done:
            print("[Plan] Task unexpectedly finished after picking drawer.")
            return

        # Step‑5  : pull drawer open (along +x – 20 cm)
        print("\n[Plan] Step‑5  pull drawer open")
        obs, reward, done = pull(env, task, pull_distance=0.20, pull_axis='x')
        if done:
            print("[Plan] Task unexpectedly finished after pull.")
            return

        # After pulling we must release the handle so the gripper can pick the
        # tomatoes later.  We simply “place” in‑place – i.e. open the gripper
        # without moving anywhere.
        print("\n[Plan] Step‑5a release drawer handle (open gripper)")
        current_pos = task.get_observation().gripper_pose[:3]
        obs, reward, done = place(env, task, current_pos, approach_distance=0.0)
        if done:
            print("[Plan] Task unexpectedly finished after releasing drawer.")
            return

        # Step‑6  : pick tomato‑1
        print("\n[Plan] Step‑6  pick tomato1 from table")
        obs, reward, done = pick(env, task, tomato1_pos, approach_axis='z')
        if done:
            print("[Plan] Task unexpectedly finished after picking tomato1.")
            return

        # Step‑7  : place tomato‑1 onto plate
        print("\n[Plan] Step‑7  place tomato1 onto plate")
        obs, reward, done = place(env, task, plate_pos, approach_axis='z')
        if done:
            print("[Plan] Task unexpectedly finished after placing tomato1.")
            return

        # Step‑8  : pick tomato‑2
        print("\n[Plan] Step‑8  pick tomato2 from table")
        obs, reward, done = pick(env, task, tomato2_pos, approach_axis='z')
        if done:
            print("[Plan] Task unexpectedly finished after picking tomato2.")
            return

        # Step‑9  : place tomato‑2 onto plate
        print("\n[Plan] Step‑9  place tomato2 onto plate")
        obs, reward, done = place(env, task, plate_pos, approach_axis='z')
        if done:
            print("[Plan] Task completed – tomatoes placed.  Reward:", reward)
        else:
            print("[Plan] Plan executed, but environment indicates ‘done = False’.")

    finally:
        # Always shut the simulator down – even if an exception was raised
        shutdown_environment(env)
        print("\n================  End of Combined‑Domain Task  =================\n")


# --------------------------------------------------------------------------- #
#                               Entrypoint                                    #
# --------------------------------------------------------------------------- #
if __name__ == "__main__":
    run_combined_task()
