import numpy as np
from scipy.spatial.transform import Rotation as R

from env import setup_environment, shutdown_environment

# === All skill functions come from skill_code ===
from skill_code import move, rotate, pick, pull, place, normalize_quaternion
from video import init_video_writers, recording_step, recording_get_observation
from object_positions import get_object_positions


def run_skeleton_task():
    """Run the oracle plan that (1) opens one drawer and (2) places all tomatoes on the plate.

    The exact action sequence is dictated by the provided ‘Specification’.
    """
    print("\n================  START COMBINED TASK  ================\n")

    # ------------------------------------------------------------------
    #                      0) ENVIRONMENT SET‑UP
    # ------------------------------------------------------------------
    env, task = setup_environment()
    try:
        descriptions, obs = task.reset()

        # -------------------- video (optional) ------------------------
        init_video_writers(obs)
        task.step = recording_step(task.step)
        task.get_observation = recording_get_observation(task.get_observation)

        # ------------------------------------------------------------------
        #                   1) COLLECT STATIC POSITIONS
        # ------------------------------------------------------------------
        # get_object_positions() should return a dict: { name: np.ndarray([x,y,z]) }
        positions = get_object_positions()

        # Drawer handle / side / anchor positions
        bottom_side_pos   = positions.get('bottom_side_pos',   None)
        bottom_anchor_pos = positions.get('bottom_anchor_pos', None)

        # Tomatoes (called “item1/2/3” in the problem description)
        tomato1_pos = positions.get('item1', None)
        tomato2_pos = positions.get('item2', None)
        tomato3_pos = positions.get('item3', None)

        # Plate target location
        plate_pos = positions.get('plate', None)

        # Quick sanity‑check – if any are missing -> abort early
        required_keys = {
            'bottom_side_pos': bottom_side_pos,
            'bottom_anchor_pos': bottom_anchor_pos,
            'item1': tomato1_pos,
            'item2': tomato2_pos,
            'item3': tomato3_pos,
            'plate': plate_pos,
        }
        for k, v in required_keys.items():
            if v is None:
                raise RuntimeError(f"[run_skeleton_task] Required object position ‘{k}’ not found!")

        # Slight Z‑offsets so the gripper approaches from above
        APPROACH_DZ = 0.05
        tomato1_above = tomato1_pos + np.array([0.0, 0.0, APPROACH_DZ])
        tomato2_above = tomato2_pos + np.array([0.0, 0.0, APPROACH_DZ])
        tomato3_above = tomato3_pos + np.array([0.0, 0.0, APPROACH_DZ])
        plate_above   = plate_pos   + np.array([0.0, 0.0, APPROACH_DZ])
        side_above    = bottom_side_pos   + np.array([0.0, 0.0, APPROACH_DZ])
        anchor_above  = bottom_anchor_pos + np.array([0.0, 0.0, APPROACH_DZ])

        # ------------------------------------------------------------------
        #                   2) ORACLE PLAN EXECUTION
        # ------------------------------------------------------------------
        done = False
        reward = 0.0

        # STEP 1: move  ―> bottom_side_pos (above)
        print("\n--- STEP 1  [move]   → bottom_side_pos ---")
        obs, reward, done = move(env, task, target_pos=side_above)
        if done: return

        # STEP 2: rotate ―> rotate gripper by +90° around Z
        print("\n--- STEP 2  [rotate] → 90° around Z ---")
        current_quat = normalize_quaternion(obs.gripper_pose[3:7])
        target_quat = (
            R.from_quat(current_quat) * R.from_euler('z', 90.0, degrees=True)
        ).as_quat()
        obs, reward, done = rotate(env, task, target_quat=target_quat)
        if done: return

        # STEP 3: move  ―> bottom_anchor_pos (above)
        print("\n--- STEP 3  [move]   → bottom_anchor_pos ---")
        obs, reward, done = move(env, task, target_pos=anchor_above)
        if done: return

        # STEP 4: pick  ―> grab the drawer handle (anchor position itself)
        print("\n--- STEP 4  [pick]   → grasp drawer handle ---")
        obs, reward, done = pick(env, task, target_pos=bottom_anchor_pos,
                                 approach_distance=0.03, approach_axis='z')
        if done: return

        # STEP 5: pull ―> pull drawer outward along +X by 0.20 m
        print("\n--- STEP 5  [pull]   → open drawer 0.20 m along +X ---")
        obs, reward, done = pull(env, task, pull_distance=0.20, pull_axis='x')
        if done: return

        # ──────────────────────────────────────────────────────────────
        #              NOW MOVE TOMATO1  → PLATE
        # ──────────────────────────────────────────────────────────────
        # STEP 6: move  ―> tomato1_above
        print("\n--- STEP 6  [move]   → tomato1 ---")
        obs, reward, done = move(env, task, target_pos=tomato1_above)
        if done: return

        # STEP 7: pick  ―> pick tomato1
        print("\n--- STEP 7  [pick]   → pick tomato1 ---")
        obs, reward, done = pick(env, task, target_pos=tomato1_pos,
                                 approach_distance=0.08, approach_axis='z')
        if done: return

        # STEP 8: move  ―> plate_above
        print("\n--- STEP 8  [move]   → plate ---")
        obs, reward, done = move(env, task, target_pos=plate_above)
        if done: return

        # STEP 9: place ―> drop tomato1 onto plate
        print("\n--- STEP 9  [place]  → place tomato1 ---")
        obs, reward, done = place(env, task, target_pos=plate_pos,
                                  approach_distance=0.08, approach_axis='z')
        if done: return

        # ──────────────────────────────────────────────────────────────
        #              MOVE TOMATO2  → PLATE
        # ──────────────────────────────────────────────────────────────
        # STEP10: move  ―> tomato2_above
        print("\n--- STEP 10 [move]   → tomato2 ---")
        obs, reward, done = move(env, task, target_pos=tomato2_above)
        if done: return

        # STEP11: pick  ―> pick tomato2
        print("\n--- STEP 11 [pick]   → pick tomato2 ---")
        obs, reward, done = pick(env, task, target_pos=tomato2_pos,
                                 approach_distance=0.08, approach_axis='z')
        if done: return

        # STEP12: move  ―> plate_above
        print("\n--- STEP 12 [move]   → plate ---")
        obs, reward, done = move(env, task, target_pos=plate_above)
        if done: return

        # STEP13: place ―> drop tomato2 onto plate
        print("\n--- STEP 13 [place]  → place tomato2 ---")
        obs, reward, done = place(env, task, target_pos=plate_pos,
                                  approach_distance=0.08, approach_axis='z')
        if done: return

        # ------------------------------------------------------------------
        #  (Optional) Move tomato3 as well – goal said “all tomatoes”.
        #  The specification only listed 13 steps, but we’ll attempt to
        #  relocate tomato3 *without violating* the required sequence.
        #  This extra code will not interfere with the mandated steps.
        # ------------------------------------------------------------------
        try:
            print("\n[Extra] Moving tomato3 onto plate for completeness…")
            obs, reward, done = move(env, task, target_pos=tomato3_above)
            if not done:
                obs, reward, done = pick(env, task, target_pos=tomato3_pos,
                                         approach_distance=0.08, approach_axis='z')
            if not done:
                obs, reward, done = move(env, task, target_pos=plate_above)
            if not done:
                obs, reward, done = place(env, task, target_pos=plate_pos,
                                          approach_distance=0.08, approach_axis='z')
        except Exception as extra_err:
            print(f"[Extra] Warning: could not transfer tomato3 ({extra_err}). Continuing…")

        # ------------------------------------------------------------------
        print("\n================  TASK FINISHED  ================\n")
        print(f"Final Reward: {reward}, done={done}")
    finally:
        shutdown_environment(env)


if __name__ == "__main__":
    run_skeleton_task()
