import numpy as np
from scipy.spatial.transform import Rotation as R

from env import setup_environment, shutdown_environment
from video import init_video_writers, recording_step, recording_get_observation
from object_positions import get_object_positions

# Import every predefined skill exactly as supplied
from skill_code import *

# ---------------------------------------------------------------------------
# Helper utilities
# ---------------------------------------------------------------------------

def safe_height(pos, dz=0.10):
    """Return a position that is dz metres above the supplied position."""
    return np.array([pos[0], pos[1], pos[2] + dz])


def execute_and_check(skill_fn, *args, **kwargs):
    """
    Wrapper that executes any skill function and aborts the entire task
    run if the underlying RLBench task reports done=True.
    """
    obs, reward, done = skill_fn(*args, **kwargs)
    if done:
        print("[Task] Environment reported done=True – terminating early.")
        raise SystemExit
    return obs


# ---------------------------------------------------------------------------
# Main task logic
# ---------------------------------------------------------------------------

def run_open_drawer_and_sort_tomatoes():
    """
    Plan (12 steps as required by specification):

        1. rotate   – align gripper to 90° (drawer‑side approach)
        2. move     – go to bottom drawer's side handle position
        3. move     – go to bottom drawer's anchor (handle) position
        4. pull     – pull drawer outwards to open it fully
        5. move     – hover above item1 (tomato 1)
        6. pick     – grasp item1
        7. move     – hover above plate
        8. place    – drop item1 on plate
        9. move     – hover above item2 (tomato 2)
       10. pick     – grasp item2
       11. move     – hover above plate
       12. place    – drop item2 on plate
    """

    print("==========  RUNNING TASK  ==========")

    # ---------------- Environment Setup ----------------
    env, task = setup_environment()
    try:
        descriptions, obs = task.reset()

        # optional video writer (kept from skeleton)
        init_video_writers(obs)
        task.step = recording_step(task.step)
        task.get_observation = recording_get_observation(task.get_observation)

        # ---------------- Gather object poses ----------------
        positions = get_object_positions()
        # Mandatory keys we rely on
        required_keys = [
            'bottom_side_pos', 'bottom_anchor_pos',
            'item1', 'item2', 'plate'
        ]
        for key in required_keys:
            if key not in positions:
                raise KeyError(f"[Task] Expected key '{key}' not in object_positions!")

        bottom_side      = np.asarray(positions['bottom_side_pos'])
        bottom_anchor    = np.asarray(positions['bottom_anchor_pos'])
        item1_pos        = np.asarray(positions['item1'])
        item2_pos        = np.asarray(positions['item2'])
        plate_pos        = np.asarray(positions['plate'])

        # ---------------- Execute plan ----------------
        # Step‑1: rotate gripper to 90 deg around z‑axis
        target_quat = R.from_euler('z', 90, degrees=True).as_quat()
        execute_and_check(
            rotate,
            env, task,
            target_quat=target_quat,
            max_steps=100,
            threshold=0.05,
            timeout=10.0
        )

        # Step‑2: move to drawer side handle position (hover)
        execute_and_check(
            move,
            env, task,
            target_pos=safe_height(bottom_side, dz=0.05)
        )

        # Step‑3: move down to drawer anchor (handle) position
        execute_and_check(
            move,
            env, task,
            target_pos=bottom_anchor
        )

        # Step‑4: pull the drawer along +x (assumption) by 20 cm
        execute_and_check(
            pull,
            env, task,
            pull_distance=0.20,
            pull_axis='x',        # change if your drawer slides along a different axis
            max_steps=100,
            threshold=0.01,
            timeout=10.0
        )

        # ---------------- Move first tomato ----------------
        # Step‑5: hover above item1
        execute_and_check(
            move,
            env, task,
            target_pos=safe_height(item1_pos, dz=0.12)
        )

        # Step‑6: pick item1
        execute_and_check(
            pick,
            env, task,
            target_pos=item1_pos,
            approach_distance=0.12
        )

        # Step‑7: hover above plate
        execute_and_check(
            move,
            env, task,
            target_pos=safe_height(plate_pos, dz=0.12)
        )

        # Step‑8: place item1 on plate
        execute_and_check(
            place,
            env, task,
            target_pos=plate_pos,
            approach_distance=0.12
        )

        # ---------------- Move second tomato ----------------
        # Step‑9: hover above item2
        execute_and_check(
            move,
            env, task,
            target_pos=safe_height(item2_pos, dz=0.12)
        )

        # Step‑10: pick item2
        execute_and_check(
            pick,
            env, task,
            target_pos=item2_pos,
            approach_distance=0.12
        )

        # Step‑11: hover above plate
        execute_and_check(
            move,
            env, task,
            target_pos=safe_height(plate_pos, dz=0.12)
        )

        # Step‑12: place item2 on plate
        execute_and_check(
            place,
            env, task,
            target_pos=plate_pos,
            approach_distance=0.12
        )

        print("[Task] All planned steps executed successfully!")

    finally:
        # Always shut down the environment cleanly
        shutdown_environment(env)
        print("==========  TASK COMPLETE  ==========")


if __name__ == "__main__":
    run_open_drawer_and_sort_tomatoes()