import numpy as np
from scipy.spatial.transform import Rotation as R
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

# All primitive skills (move, pick, place, rotate, pull) are imported here
from skill_code import *

from video import init_video_writers, recording_step, recording_get_observation
from object_positions import get_object_positions


def _safe_get_pos(name: str, positions: dict):
    """Utility: fetch object position, raise helpful error if not found."""
    if name not in positions:
        raise KeyError(f"[run_task] Position for '{name}' not found in object_positions.")
    return positions[name]


def run_skeleton_task():
    """Execute the oracle plan to open a drawer and move two tomatoes to the plate."""
    print("===== Starting Skeleton Task =====")

    # ------------------------------------------------------------------
    # Environment setup
    # ------------------------------------------------------------------
    env, task = setup_environment()
    try:
        # Reset task and obtain initial observation
        descriptions, obs = task.reset()

        # (Optional) video capture
        init_video_writers(obs)
        # Wrap step & get_observation so video captures automatically
        task.step = recording_step(task.step)
        task.get_observation = recording_get_observation(task.get_observation)

        # ------------------------------------------------------------------
        # Retrieve positions of all relevant objects
        # ------------------------------------------------------------------
        positions = get_object_positions()

        # Drawer related positions
        bottom_side_pos   = _safe_get_pos("bottom_side_pos",   positions)
        bottom_anchor_pos = _safe_get_pos("bottom_anchor_pos", positions)
        waypoint_safe     = positions.get("waypoint1", bottom_side_pos)

        # Items & plate
        item1_pos = _safe_get_pos("item1",  positions)
        item2_pos = _safe_get_pos("item2",  positions)
        plate_pos = _safe_get_pos("plate",  positions)

        # ------------------------------------------------------------------
        # Begin Oracle Plan (10 steps as per specification)
        # ------------------------------------------------------------------
        print("\n===== [Plan] Step 1: move gripper to bottom_side_pos =====")
        obs, reward, done = move(env, task, target_pos=bottom_side_pos)
        if done:
            print("[Task] Ended unexpectedly after step 1.")
            return

        print("\n===== [Plan] Step 2: rotate gripper to 90 deg =====")
        quat_ninety = R.from_euler('xyz', [0, 0, np.deg2rad(90)]).as_quat()  # xyzw
        obs, reward, done = rotate(env, task, target_quat=quat_ninety)
        if done:
            print("[Task] Ended unexpectedly after step 2.")
            return

        print("\n===== [Plan] Step 3: move gripper to bottom_anchor_pos =====")
        obs, reward, done = move(env, task, target_pos=bottom_anchor_pos)
        if done:
            print("[Task] Ended unexpectedly after step 3.")
            return

        print("\n===== [Plan] Step 4: pick the drawer handle (bottom) =====")
        obs, reward, done = pick(env, task, target_pos=bottom_anchor_pos)
        if done:
            print("[Task] Ended unexpectedly after step 4.")
            return

        print("\n===== [Plan] Step 5: pull drawer open =====")
        # Pull outwards along +x axis by 0.10 m (adjust if drawer opens elsewhere)
        obs, reward, done = pull(env, task, pull_distance=0.10, pull_axis='x')
        if done:
            print("[Task] Ended unexpectedly after step 5.")
            return

        print("\n===== [Plan] Step 6: move gripper back to safe position =====")
        obs, reward, done = move(env, task, target_pos=waypoint_safe)
        if done:
            print("[Task] Ended unexpectedly after step 6.")
            return

        # ---------- Tomato 1 ----------
        print("\n===== [Plan] Step 7: pick tomato/item1 from table =====")
        obs, reward, done = pick(env, task, target_pos=item1_pos)
        if done:
            print("[Task] Ended unexpectedly after picking item1.")
            return

        print("\n===== [Plan] Step 8: place tomato/item1 onto plate =====")
        obs, reward, done = place(env, task, target_pos=plate_pos)
        if done:
            print("[Task] Ended unexpectedly after placing item1.")
            return

        # ---------- Tomato 2 ----------
        print("\n===== [Plan] Step 9: pick tomato/item2 from table =====")
        obs, reward, done = pick(env, task, target_pos=item2_pos)
        if done:
            print("[Task] Ended unexpectedly after picking item2.")
            return

        print("\n===== [Plan] Step 10: place tomato/item2 onto plate =====")
        obs, reward, done = place(env, task, target_pos=plate_pos)
        # Task ideally completes here
        if done:
            print("\n[Task] Plan executed successfully! Reward:", reward)
        else:
            print("\n[Task] Plan finished; done flag not set (this may be expected).")

    except Exception as e:
        print("[run_task] Exception encountered:", e)
        raise
    finally:
        # Always shut down environment properly
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()
