# === Auto‑Generated Task Runner ===
#
# This script follows the skeleton pattern that is shipped with the
# evaluation environment.  It “fills in the blanks” with a concrete
# sequence of skill calls that matches the oracle plan given in the
# specification:
#
#   1)  rotate  gripper zero‑>ninety
#   2)  move    nowhere‑pos     -> side‑pos‑bottom
#   3)  move    side‑pos‑bottom -> anchor‑pos‑bottom
#   4)  pick    drawer handle   (grasp the anchor position)
#   5)  pull    drawer          (open)
#   6)  pick    tomato1
#   7)  place   tomato1 on plate
#   8)  pick    tomato2
#   9)  place   tomato2 on plate
#
# No new primitive skills are introduced – only the predefined skills
#   rotate, move, pick, pull, place
# are called.

import numpy as np
from scipy.spatial.transform import Rotation as R

from env import setup_environment, shutdown_environment

# === predefined helpers (video, skill, etc.) ===
from video import (                     # pylint: disable=import-error
    init_video_writers,
    recording_step,
    recording_get_observation,
)
from object_positions import get_object_positions  # pylint: disable=import-error
from skill_code import *                            # noqa: F403 F401


def _quat_from_euler_xyz(deg_x: float, deg_y: float, deg_z: float):
    """Utility – returns xyzw quaternion from XYZ euler degrees."""
    return R.from_euler("xyz", [deg_x, deg_y, deg_z], degrees=True).as_quat()


def run_skeleton_task():
    print("===== Starting Skeleton Task =====")

    # ------------------------------------------------------------------
    # Environment set‑up
    # ------------------------------------------------------------------
    env, task = setup_environment()
    try:
        descriptions, obs = task.reset()

        # optional video writer / recorder
        init_video_writers(obs)
        task.step = recording_step(task.step)                   # wrap
        task.get_observation = recording_get_observation(       # wrap
            task.get_observation
        )

        # ------------------------------------------------------------------
        # Fetch all relevant positions (returns {name: np.ndarray([x,y,z])})
        # ------------------------------------------------------------------
        positions = get_object_positions()

        bottom_side_pos = positions["bottom_side_pos"]
        bottom_anchor_pos = positions["bottom_anchor_pos"]

        tomato1_pos = positions["tomato1"]
        tomato2_pos = positions["tomato2"]
        plate_pos = positions["plate"]

        # ------------------------------------------------------------------
        # 1) ROTATE – zero_deg → ninety_deg
        # ------------------------------------------------------------------
        # Assume “zero_deg” is identity, and “ninety_deg” is +90° about Z.
        target_quat = _quat_from_euler_xyz(0, 0, 90)
        print("\n[PLAN 1] rotate gripper to ninety_deg")
        obs, reward, done = rotate(env, task, target_quat)      # noqa: F405
        if done:
            print("[Abort] Task ended during rotate.")
            return

        # ------------------------------------------------------------------
        # 2) MOVE to side‑pos‑bottom
        # ------------------------------------------------------------------
        print("\n[PLAN 2] move to bottom_side_pos:", bottom_side_pos)
        obs, reward, done = move(env, task, bottom_side_pos)    # noqa: F405
        if done:
            print("[Abort] Task ended during move‑to‑side.")
            return

        # ------------------------------------------------------------------
        # 3) MOVE to anchor‑pos‑bottom
        # ------------------------------------------------------------------
        print("\n[PLAN 3] move to bottom_anchor_pos:", bottom_anchor_pos)
        obs, reward, done = move(env, task, bottom_anchor_pos)  # noqa: F405
        if done:
            print("[Abort] Task ended during move‑to‑anchor.")
            return

        # ------------------------------------------------------------------
        # 4) PICK the drawer handle (grasp)
        # ------------------------------------------------------------------
        print("\n[PLAN 4] pick drawer handle (anchor position)")
        obs, reward, done = pick(                               # noqa: F405
            env,
            task,
            target_pos=bottom_anchor_pos,
            approach_distance=0.07,         # short approach for handle
            approach_axis="-y",             # approach from +y toward -y
        )
        if done:
            print("[Abort] Task ended during pick‑drawer.")
            return

        # ------------------------------------------------------------------
        # 5) PULL to open the drawer
        #    – pull along -y (backwards) by ~0.15 m
        # ------------------------------------------------------------------
        print("\n[PLAN 5] pull drawer outwards")
        obs, reward, done = pull(                               # noqa: F405
            env,
            task,
            pull_distance=0.15,
            pull_axis="-y",
        )
        if done:
            print("[Abort] Task ended during pull.")
            return

        # ------------------------------------------------------------------
        # 6) PICK tomato1
        # ------------------------------------------------------------------
        print("\n[PLAN 6] pick tomato1 at", tomato1_pos)
        obs, reward, done = pick(
            env,
            task,
            target_pos=tomato1_pos,
            approach_distance=0.15,
            approach_axis="z",
        )
        if done:
            print("[Abort] Task ended during pick tomato1.")
            return

        # ------------------------------------------------------------------
        # 7) PLACE tomato1 on plate
        # ------------------------------------------------------------------
        print("\n[PLAN 7] place tomato1 on plate at", plate_pos)
        obs, reward, done = place(
            env,
            task,
            target_pos=plate_pos,
            approach_distance=0.15,
            approach_axis="z",
        )
        if done:
            print("[Abort] Task ended during place tomato1.")
            return

        # ------------------------------------------------------------------
        # 8) PICK tomato2
        # ------------------------------------------------------------------
        print("\n[PLAN 8] pick tomato2 at", tomato2_pos)
        obs, reward, done = pick(
            env,
            task,
            target_pos=tomato2_pos,
            approach_distance=0.15,
            approach_axis="z",
        )
        if done:
            print("[Abort] Task ended during pick tomato2.")
            return

        # ------------------------------------------------------------------
        # 9) PLACE tomato2 on plate
        # ------------------------------------------------------------------
        print("\n[PLAN 9] place tomato2 on plate at", plate_pos)
        obs, reward, done = place(
            env,
            task,
            target_pos=plate_pos,
            approach_distance=0.15,
            approach_axis="z",
        )

        # ------------------------------------------------------------------
        # RESULT
        # ------------------------------------------------------------------
        if done:
            print("\n=== Task finished by environment (done=True). Reward:", reward)
        else:
            print("\n=== Plan executed.  Environment not finished yet (done=False).")

    finally:
        # ------------------------------------------------------------------
        # Clean shutdown always!
        # ------------------------------------------------------------------
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()
