import numpy as np

from env import setup_environment, shutdown_environment
from skill_code import rotate, move, pick, pull, place
from video import init_video_writers, recording_step, recording_get_observation
from object_positions import get_object_positions


def run_skeleton_task():
    """Execute the oracle plan:
         1) Rotate the gripper 90 deg
         2) Move to the side‑handle of the *bottom* drawer
         3) Move from the side‑handle to the anchor‑handle
         4) Grab the drawer handle
         5) Pull the drawer open
         6) Pick tomato1 –> place on plate
         7) Pick tomato2 –> place on plate
    """
    print("===== Starting Combined‑Domain Task =====")
    env, task = setup_environment()
    try:
        # ————————————————————————————————————————————————
        #  Environment / video initialisation
        # ————————————————————————————————————————————————
        descriptions, obs = task.reset()
        init_video_writers(obs)

        orig_step = task.step
        task.step = recording_step(orig_step)
        orig_get_obs = task.get_observation
        task.get_observation = recording_get_observation(orig_get_obs)

        # ————————————————————————————————————————————————
        #  Fetch object poses that we will need
        # ————————————————————————————————————————————————
        pos = get_object_positions()          # Dict: name → np.ndarray([x,y,z])
        try:
            side_handle = pos['bottom_side_pos']
            anchor_handle = pos['bottom_anchor_pos']
            tomato1_pos = pos['tomato1']
            tomato2_pos = pos['tomato2']
            plate_pos = pos['plate']
        except KeyError as e:
            raise RuntimeError(f"Missing expected object in position dictionary: {e}")

        # Safety‑copy the reward/done vars used below
        reward, done = 0.0, False

        # ------------------------------------------------------------------
        #  STEP‑1  Rotate gripper to 90 deg (about world‑Z)
        # ------------------------------------------------------------------
        target_quat = np.array([0, 0, 0.70710678, 0.70710678])  # (x,y,z,w) for +90° yaw
        obs, reward, done = rotate(env, task, target_quat)
        if done:
            print("[Task] Finished unexpectedly after rotate.")
            return

        # ------------------------------------------------------------------
        #  STEP‑2  Move to drawer’s side position
        # ------------------------------------------------------------------
        obs, reward, done = move(env, task, side_handle)
        if done:
            print("[Task] Finished unexpectedly after move‑to‑side.")
            return

        # ------------------------------------------------------------------
        #  STEP‑3  Move from side‑pos to anchor‑pos
        # ------------------------------------------------------------------
        obs, reward, done = move(env, task, anchor_handle)
        if done:
            print("[Task] Finished unexpectedly after move‑to‑anchor.")
            return

        # ------------------------------------------------------------------
        #  STEP‑4  Pick (grab) the drawer handle
        # ------------------------------------------------------------------
        obs, reward, done = pick(env, task, anchor_handle, approach_axis='z')
        if done:
            print("[Task] Finished unexpectedly after pick‑drawer.")
            return

        # ------------------------------------------------------------------
        #  STEP‑5  Pull drawer open (pull 0.20 m along +x)
        # ------------------------------------------------------------------
        obs, reward, done = pull(env, task, pull_distance=0.20, pull_axis='x')
        if done:
            print("[Task] Finished unexpectedly after pull.")
            return

        # ------------------------------------------------------------------
        #  STEP‑6  Pick tomato1 off table
        # ------------------------------------------------------------------
        obs, reward, done = pick(env, task, tomato1_pos, approach_axis='z')
        if done:
            print("[Task] Finished unexpectedly after pick tomato1.")
            return

        # ------------------------------------------------------------------
        #  STEP‑7  Place tomato1 on plate
        # ------------------------------------------------------------------
        obs, reward, done = place(env, task, plate_pos, approach_axis='z')
        if done:
            print("[Task] Finished unexpectedly after place tomato1.")
            return

        # ------------------------------------------------------------------
        #  STEP‑8  Pick tomato2 off table
        # ------------------------------------------------------------------
        obs, reward, done = pick(env, task, tomato2_pos, approach_axis='z')
        if done:
            print("[Task] Finished unexpectedly after pick tomato2.")
            return

        # ------------------------------------------------------------------
        #  STEP‑9  Place tomato2 on plate
        # ------------------------------------------------------------------
        obs, reward, done = place(env, task, plate_pos, approach_axis='z')
        if done:
            print("[Task] Task completed successfully! Reward:", reward)
        else:
            print("[Task] Task finished (done=False) – goal likely reached.")

    finally:
        shutdown_environment(env)
        print("===== End of Combined‑Domain Task =====")


if __name__ == "__main__":
    run_skeleton_task()
