# run_combined_task.py

import numpy as np
import math
import traceback

from env import setup_environment, shutdown_environment
from video import init_video_writers, recording_step, recording_get_observation
from object_positions import get_object_positions

# ====== Import the predefined skills exactly as provided ======
from skill_code import pick, place, move, rotate, pull


def run_combined_task():
    """
    Executes the oracle plan supplied in the specification:

        1. rotate(gripper, zero_deg, ninety_deg)
        2. move‑to‑side   (≈ move)   → bottom_side_pos
        3. move‑to‑anchor (≈ move)   → bottom_anchor_pos
        4. pick‑drawer    (≈ pick)   → bottom_anchor_pos
        5. pull(drawer)               → pull the drawer open
        6. pick(tomato1, table)
        7. place(tomato1, plate)
        8. pick(tomato2, table)
        9. place(tomato2, plate)

    Only the predefined skills contained in `skill_code` are used.
    """

    print("===== [Task]  START  =====")

    # ------------------------------------------------------------------
    # 1) Environment setup
    # ------------------------------------------------------------------
    env, task = setup_environment()
    try:
        # Reset the RLBench task
        descriptions, obs = task.reset()

        # Optional: start video recording
        init_video_writers(obs)
        task.step = recording_step(task.step)
        task.get_observation = recording_get_observation(task.get_observation)

        # ------------------------------------------------------------------
        # 2) Retrieve all relevant object positions from the scene
        # ------------------------------------------------------------------
        positions = get_object_positions()

        # Drawer‑related waypoints
        bottom_side_pos   = np.asarray(positions['bottom_side_pos'],   dtype=np.float32)
        bottom_anchor_pos = np.asarray(positions['bottom_anchor_pos'], dtype=np.float32)

        # Tomatoes & plate
        tomato1_pos = np.asarray(positions['tomato1'], dtype=np.float32)
        tomato2_pos = np.asarray(positions['tomato2'], dtype=np.float32)
        plate_pos   = np.asarray(positions['plate'],   dtype=np.float32)

        # ------------------------------------------------------------------
        # 3) Execute the oracle plan (steps 1‑9)
        # ------------------------------------------------------------------
        done = False
        reward = 0.0

        # ---- STEP 1 : rotate(gripper, zero_deg → ninety_deg) ----------
        # Create a quaternion representing a 90‑degree rotation about the Z‑axis
        ninety_deg_quat_xyzw = np.asarray(
            (0.0, 0.0, math.sin(math.radians(90.0) / 2.0), math.cos(math.radians(90.0) / 2.0)),
            dtype=np.float32,
        )
        print("\n[PLAN] Step 1 : rotate gripper 90 deg about Z")
        obs, reward, done = rotate(env, task, target_quat=ninety_deg_quat_xyzw)
        if done:
            print("[PLAN] Terminated after step 1")
            return

        # ---- STEP 2 : move‑to‑side  (≈ move) --------------------------
        print("\n[PLAN] Step 2 : move to bottom_side_pos =", bottom_side_pos)
        obs, reward, done = move(env, task, target_pos=bottom_side_pos)
        if done:
            print("[PLAN] Terminated after step 2")
            return

        # ---- STEP 3 : move‑to‑anchor (≈ move) -------------------------
        print("\n[PLAN] Step 3 : move to bottom_anchor_pos =", bottom_anchor_pos)
        obs, reward, done = move(env, task, target_pos=bottom_anchor_pos)
        if done:
            print("[PLAN] Terminated after step 3")
            return

        # ---- STEP 4 : pick‑drawer (≈ pick) ----------------------------
        print("\n[PLAN] Step 4 : grip the drawer handle (bottom_anchor_pos)")
        obs, reward, done = pick(
            env,
            task,
            target_pos=bottom_anchor_pos,
            approach_distance=0.10,
            approach_axis='z',
        )
        if done:
            print("[PLAN] Terminated after step 4")
            return

        # ---- STEP 5 : pull the drawer open ----------------------------
        # Pull 0.20 m along the X‑axis (positive direction)
        print("\n[PLAN] Step 5 : pull the drawer outwards (0.20 m on +X)")
        obs, reward, done = pull(
            env,
            task,
            pull_distance=0.20,
            pull_axis='x',
        )
        if done:
            print("[PLAN] Terminated after step 5")
            return

        # ---- STEP 6 : pick tomato1 from table -------------------------
        print("\n[PLAN] Step 6 : pick tomato1")
        obs, reward, done = pick(
            env,
            task,
            target_pos=tomato1_pos,
            approach_distance=0.10,
            approach_axis='z',
        )
        if done:
            print("[PLAN] Terminated after step 6")
            return

        # ---- STEP 7 : place tomato1 on plate --------------------------
        print("\n[PLAN] Step 7 : place tomato1 on plate")
        obs, reward, done = place(
            env,
            task,
            target_pos=plate_pos,
            approach_distance=0.12,
            approach_axis='z',
        )
        if done:
            print("[PLAN] Terminated after step 7")
            return

        # ---- STEP 8 : pick tomato2 from table -------------------------
        print("\n[PLAN] Step 8 : pick tomato2")
        obs, reward, done = pick(
            env,
            task,
            target_pos=tomato2_pos,
            approach_distance=0.10,
            approach_axis='z',
        )
        if done:
            print("[PLAN] Terminated after step 8")
            return

        # ---- STEP 9 : place tomato2 on plate --------------------------
        print("\n[PLAN] Step 9 : place tomato2 on plate")
        obs, reward, done = place(
            env,
            task,
            target_pos=plate_pos,
            approach_distance=0.12,
            approach_axis='z',
        )
        if done:
            print("[PLAN] Finished after step 9 — Goal reached!")
        else:
            print("[PLAN] All plan steps executed — task may still be running (done=False).")

    except Exception as e:
        print("==========  EXCEPTION OCCURRED  ==========")
        traceback.print_exc()
    finally:
        # ------------------------------------------------------------------
        # 4) Always shutdown the environment
        # ------------------------------------------------------------------
        shutdown_environment(env)
        print("===== [Task]  END  =====")


if __name__ == "__main__":
    run_combined_task()
