# run_combined_task.py

# Description:
#   Executes the oracle plan given in the specification.
#   The plan (in plain language) is:
#        1) Rotate the gripper 90 deg
#        2) Move to the drawer side‑pos
#        3) Move to the drawer anchor‑pos
#        4) Grasp the drawer handle
#        5) Pull (slide) the drawer open
#        6) Pick tomato1 from the table and place it on the plate
#        7) Pick tomato2 from the table and place it on the plate
#
#   Only the pre‑implemented skills contained in `skill_code.py`
#   (rotate, move, pull, pick, place) are used.

import sys
import traceback
import numpy as np
from scipy.spatial.transform import Rotation as R
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment
from skill_code import *   # rotate, move, pull, pick, place, …
from video import init_video_writers, recording_step, recording_get_observation
from object_positions import get_object_positions


# -----------------------------------------------------------
# Helper utilities
# -----------------------------------------------------------

def safe_skill_call(skill_fn, *args, **kwargs):
    """Invoke a skill and gracefully handle exceptions."""
    try:
        obs, reward, done = skill_fn(*args, **kwargs)
        return obs, reward, done
    except Exception as exc:   # noqa: broad‑except (intentional for robustness)
        print("[ERROR]", skill_fn.__name__, "raised an exception:")
        traceback.print_exc(file=sys.stdout)
        raise RuntimeError(f"Skill {skill_fn.__name__} failed!") from exc


def rotate_90_deg_about_z(obs):
    """Return a quaternion representing +90° rotation about Z from current pose."""
    current_quat = obs.gripper_pose[3:7]          # xyzw
    current_rot = R.from_quat(current_quat)
    delta_rot = R.from_euler('z', 90, degrees=True)
    target_quat = (delta_rot * current_rot).as_quat()
    return target_quat


# -----------------------------------------------------------
# Main task runner
# -----------------------------------------------------------

def run_combined_task():
    print("===== Starting Combined Drawer‑and‑Tomato Task =====")

    # === Environment ========================================================
    env, task = setup_environment()
    try:
        descriptions, obs = task.reset()

        # (optional) video recording helpers
        init_video_writers(obs)
        task.step = recording_step(task.step)
        task.get_observation = recording_get_observation(task.get_observation)

        # === Retrieve all relevant object positions ========================
        positions = get_object_positions()
        required_keys = [
            'bottom_side_pos', 'bottom_anchor_pos',
            'tomato1', 'tomato2', 'plate'
        ]
        for k in required_keys:
            if k not in positions:
                raise KeyError(f"[object_positions] Missing key: {k}")

        # Convenience shorthands
        p_side   = np.asarray(positions['bottom_side_pos'])
        p_anchor = np.asarray(positions['bottom_anchor_pos'])
        p_t1     = np.asarray(positions['tomato1'])
        p_t2     = np.asarray(positions['tomato2'])
        p_plate  = np.asarray(positions['plate'])

        # === Execute Oracle Plan ===========================================

        # Step‑1: rotate gripper (zero_deg ➜ ninety_deg)
        print("\n[Plan‑1] rotate gripper 90 deg …")
        target_quat = rotate_90_deg_about_z(task.get_observation())
        obs, reward, done = safe_skill_call(
            rotate, env, task, target_quat,
            max_steps=120, threshold=0.03, timeout=10.0
        )
        if done:
            print("[Early‑Done] Task finished unexpectedly after rotate.")
            return

        # Step‑2: move‑to‑side (nowhere‑pos ➜ side‑pos‑bottom)
        print("\n[Plan‑2] move to drawer side position …")
        obs, reward, done = safe_skill_call(
            move, env, task, p_side,
            max_steps=120, threshold=0.01, timeout=10.0
        )
        if done:
            print("[Early‑Done] Task finished unexpectedly after move‑to‑side.")
            return

        # Step‑3: move‑to‑anchor (side‑pos‑bottom ➜ anchor‑pos‑bottom)
        print("\n[Plan‑3] move to drawer anchor position …")
        obs, reward, done = safe_skill_call(
            move, env, task, p_anchor,
            max_steps=120, threshold=0.01, timeout=10.0
        )
        if done:
            print("[Early‑Done] Task finished unexpectedly after move‑to‑anchor.")
            return

        # Step‑4: pick‑drawer (grasp handle)
        print("\n[Plan‑4] grasp drawer handle …")
        obs, reward, done = safe_skill_call(
            pick, env, task, target_pos=p_anchor,
            approach_distance=0.12, approach_axis='z',
            max_steps=120, threshold=0.01, timeout=10.0
        )
        if done:
            print("[Early‑Done] Task finished unexpectedly after pick‑drawer.")
            return

        # Step‑5: pull (slide drawer open)
        print("\n[Plan‑5] pull drawer …")
        obs, reward, done = safe_skill_call(
            pull, env, task,
            pull_distance=0.20, pull_axis='x',
            max_steps=120, threshold=0.01, timeout=10.0
        )
        if done:
            print("[Early‑Done] Task finished unexpectedly after pull.")
            return

        # Step‑6: pick tomato1
        print("\n[Plan‑6] pick tomato1 …")
        obs, reward, done = safe_skill_call(
            pick, env, task, target_pos=p_t1,
            approach_distance=0.15, approach_axis='z',
            max_steps=120, threshold=0.01, timeout=10.0
        )
        if done:
            print("[Early‑Done] Task finished unexpectedly after picking tomato1.")
            return

        # Step‑7: place tomato1 ➜ plate
        print("\n[Plan‑7] place tomato1 onto plate …")
        obs, reward, done = safe_skill_call(
            place, env, task, target_pos=p_plate,
            approach_distance=0.15, approach_axis='z',
            max_steps=120, threshold=0.01, timeout=10.0
        )
        if done:
            print("[Early‑Done] Task finished unexpectedly after placing tomato1.")
            return

        # Step‑8: pick tomato2
        print("\n[Plan‑8] pick tomato2 …")
        obs, reward, done = safe_skill_call(
            pick, env, task, target_pos=p_t2,
            approach_distance=0.15, approach_axis='z',
            max_steps=120, threshold=0.01, timeout=10.0
        )
        if done:
            print("[Early‑Done] Task finished unexpectedly after picking tomato2.")
            return

        # Step‑9: place tomato2 ➜ plate
        print("\n[Plan‑9] place tomato2 onto plate …")
        obs, reward, done = safe_skill_call(
            place, env, task, target_pos=p_plate,
            approach_distance=0.15, approach_axis='z',
            max_steps=120, threshold=0.01, timeout=10.0
        )

        # === Finalisation ===================================================
        if done:
            print("\n===== SUCCESS: Goal achieved, task reports done=True! =====")
        else:
            print("\n===== Finished plan. 'done' flag is False; "
                  "but the plan executed completely. =====")

    finally:
        shutdown_environment(env)

    print("===== End of Combined Task =====\n")


if __name__ == "__main__":
    run_combined_task()
