import numpy as np
from scipy.spatial.transform import Rotation as R

from env import setup_environment, shutdown_environment
from video import init_video_writers, recording_step, recording_get_observation
from object_positions import get_object_positions

# === Pre-implemented skills (DO NOT REDEFINE) =============================
from skill_code import rotate, move, pick, pull, place
# ==========================================================================


def run_oracle_plan():
    """
    Executes the oracle plan that
      1) opens the (bottom) drawer and
      2) transfers exactly the two tomatoes onto the plate
    using only the skills required by the specification.
    """
    print("\n==========  [Task]  START  ==========")

    # ----------------------------------------------------------------------
    # 0) Environment initialisation
    # ----------------------------------------------------------------------
    env, task = setup_environment()
    try:
        _, obs = task.reset()

        # Optional video recording hooks
        init_video_writers(obs)
        task.step            = recording_step(task.step)
        task.get_observation = recording_get_observation(task.get_observation)

        # ------------------------------------------------------------------
        # Retrieve all object positions that are required by the plan
        # ------------------------------------------------------------------
        positions = get_object_positions()

        # Check for drawer-related positions
        required_keys = ["bottom_side_pos", "bottom_anchor_pos", "plate"]
        missing = [k for k in required_keys if k not in positions]
        if missing:
            raise KeyError(f"[Task] Missing required keys in object_positions: {missing}")

        # Resolve tomato keys (support both naming conventions)
        def resolve_tomato(candidates, pretty_name):
            for key in candidates:
                if key in positions:
                    return key
            raise KeyError(f"[Task] Could not find {pretty_name} in object_positions")

        tomato1_key = resolve_tomato(["tomato1", "item1"], "tomato-1")
        tomato2_key = resolve_tomato(["tomato2", "item2"], "tomato-2")

        # Convert to numpy arrays for convenience
        bottom_side_pos   = np.asarray(positions["bottom_side_pos"],   dtype=float)
        bottom_anchor_pos = np.asarray(positions["bottom_anchor_pos"], dtype=float)
        plate_pos         = np.asarray(positions["plate"],             dtype=float)
        tomato1_pos       = np.asarray(positions[tomato1_key],         dtype=float)
        tomato2_pos       = np.asarray(positions[tomato2_key],         dtype=float)

        # ------------------------------------------------------------------
        # STEP 1  – rotate  (0° ➜ 90° about Z-axis)
        # ------------------------------------------------------------------
        ninety_deg_quat = R.from_euler("xyz", [0.0, 0.0, np.pi / 2]).as_quat()
        obs, reward, done = rotate(env, task, target_quat=ninety_deg_quat)
        if done:
            print("[Task] Finished early at step-1 (rotate).")
            return

        # ------------------------------------------------------------------
        # STEP 2  – move-to-side  (initial ➜ drawer side position)
        # ------------------------------------------------------------------
        obs, reward, done = move(env, task, target_pos=bottom_side_pos)
        if done:
            print("[Task] Finished early at step-2 (move-to-side).")
            return

        # ------------------------------------------------------------------
        # STEP 3  – move-to-anchor  (side ➜ drawer anchor/handle)
        # ------------------------------------------------------------------
        obs, reward, done = move(env, task, target_pos=bottom_anchor_pos)
        if done:
            print("[Task] Finished early at step-3 (move-to-anchor).")
            return

        # ------------------------------------------------------------------
        # STEP 4  – pick-drawer  (grasp the drawer handle, approach from −Z)
        # ------------------------------------------------------------------
        obs, reward, done = pick(
            env,
            task,
            target_pos=bottom_anchor_pos,
            approach_axis='-z'        # horizontal approach onto the handle
        )
        if done:
            print("[Task] Finished early at step-4 (pick-drawer).")
            return

        # ------------------------------------------------------------------
        # STEP 5  – pull  (open the drawer along +X by 20 cm)
        # ------------------------------------------------------------------
        obs, reward, done = pull(
            env,
            task,
            pull_distance=0.20,        # 20 cm
            pull_axis='x'
        )
        if done:
            print("[Task] Finished early at step-5 (pull).")
            return

        # ------------------------------------------------------------------
        # STEP 6  – pick tomato-1  (approach from −Z)
        # ------------------------------------------------------------------
        obs, reward, done = move(env, task, target_pos=tomato1_pos)
        if done:
            print("[Task] Finished early at intermediate move before step-6.")
            return

        obs, reward, done = pick(
            env,
            task,
            target_pos=tomato1_pos,
            approach_axis='-z'
        )
        if done:
            print("[Task] Finished early at step-6 (pick tomato-1).")
            return

        # ------------------------------------------------------------------
        # STEP 7  – place tomato-1 on plate
        # ------------------------------------------------------------------
        obs, reward, done = move(env, task, target_pos=plate_pos)
        if done:
            print("[Task] Finished early at intermediate move before step-7.")
            return

        obs, reward, done = place(
            env,
            task,
            target_pos=plate_pos,
            approach_axis='-z'
        )
        if done:
            print("[Task] Finished early at step-7 (place tomato-1).")
            return

        # ------------------------------------------------------------------
        # STEP 8  – pick tomato-2
        # ------------------------------------------------------------------
        obs, reward, done = move(env, task, target_pos=tomato2_pos)
        if done:
            print("[Task] Finished early at intermediate move before step-8.")
            return

        obs, reward, done = pick(
            env,
            task,
            target_pos=tomato2_pos,
            approach_axis='-z'
        )
        if done:
            print("[Task] Finished early at step-8 (pick tomato-2).")
            return

        # ------------------------------------------------------------------
        # STEP 9  – place tomato-2 on plate
        # ------------------------------------------------------------------
        obs, reward, done = move(env, task, target_pos=plate_pos)
        if done:
            print("[Task] Finished early at intermediate move before step-9.")
            return

        obs, reward, done = place(
            env,
            task,
            target_pos=plate_pos,
            approach_axis='-z'
        )

        # ------------------------------------------------------------------
        #  Result
        # ------------------------------------------------------------------
        if done:
            print("[Task] Successfully completed all steps – Goal achieved!")
        else:
            print("[Task] Plan executed but task reports done=False. "
                  "Verify success conditions inside the RLBench task.")

    except Exception as exc:
        print(f"[Task] Exception caught: {exc}")

    finally:
        shutdown_environment(env)
        print("==========  [Task]   END   ==========\n")


if __name__ == "__main__":
    run_oracle_plan()