# run_open_drawer_and_place_tomatoes.py
#
# This file completes the generic skeleton by instantiating the oracle
# plan given in the specification.  Only the predefined skills that
# already exist in `skill_code.py` are used – nothing new is introduced.
# The code:
#   1) starts the simulation,
#   2) executes the nine-step oracle plan (rotate → drawer manipulation
#      → pick / place tomatoes),
#   3) gracefully shuts everything down.

import numpy as np
from scipy.spatial.transform import Rotation as R
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment
from video import init_video_writers, recording_step, recording_get_observation
from object_positions import get_object_positions

# === PRE-DEFINED SKILLS (imported exactly as implemented elsewhere) ===
from skill_code import rotate, move, pull, pick, place


def _quat_from_angle_name(angle_name: str) -> np.ndarray:
    """Utility: map the symbolic ‘zero_deg’ / ‘ninety_deg’ names that appear
    in the PDDL domain to real XYZW quaternions used by the rotate skill."""
    if angle_name == "zero_deg":
        # Identity rotation
        return R.from_euler('xyz', [0, 0, 0], degrees=True).as_quat()
    elif angle_name == "ninety_deg":
        # Rotate 90° about +Z (common for drawer handles mounted on X-axis)
        return R.from_euler('z', 90, degrees=True).as_quat()
    else:
        raise ValueError(f"Unknown angle literal: {angle_name}")


def run_open_drawer_and_place_tomatoes():
    print("=====  Task: open drawer + place tomatoes  =====")

    env, task = setup_environment()
    try:
        # ------------------------------------------------------------------
        # 1.  Reset the task / initialise recording
        # ------------------------------------------------------------------
        _, obs = task.reset()
        init_video_writers(obs)

        # Wrap task.step / task.get_observation so that every call is recorded
        task.step = recording_step(task.step)
        task.get_observation = recording_get_observation(task.get_observation)

        # ------------------------------------------------------------------
        # 2.  Pre-compute all object positions we will need
        # ------------------------------------------------------------------
        # Expected keys (see provided object list)
        expected_keys = [
            "bottom_side_pos", "bottom_anchor_pos",
            "tomato1", "tomato2", "plate"
        ]
        positions = get_object_positions()
        missing = [k for k in expected_keys if k not in positions]
        if missing:
            raise RuntimeError(f"Missing keys from object_positions: {missing}")

        # Short-hand variables
        bottom_side = positions["bottom_side_pos"]
        bottom_anchor = positions["bottom_anchor_pos"]
        tomato1_pos = positions["tomato1"]
        tomato2_pos = positions["tomato2"]
        plate_pos = positions["plate"]

        # ------------------------------------------------------------------
        # 3.  Execute the oracle plan step-by-step
        # ------------------------------------------------------------------
        done = False
        reward = 0.0

        # ----- Step-1: rotate(gripper, zero_deg, ninety_deg)
        target_quat = _quat_from_angle_name("ninety_deg")
        obs, reward, done = rotate(env, task, target_quat)
        if done:
            print("[Plan] Terminated after rotate.")
            return

        # ----- Step-2: move-to-side(gripper, bottom, nowhere-pos, side-pos-bottom)
        obs, reward, done = move(env, task, bottom_side)
        if done:
            print("[Plan] Terminated after move-to-side.")
            return

        # ----- Step-3: move-to-anchor(gripper, bottom, side-pos-bottom, anchor-pos-bottom)
        obs, reward, done = move(env, task, bottom_anchor)
        if done:
            print("[Plan] Terminated after move-to-anchor.")
            return

        # ----- Step-4: pick-drawer(gripper, bottom, anchor-pos-bottom)
        obs, reward, done = pick(env, task, bottom_anchor, approach_distance=0.05)
        if done:
            print("[Plan] Terminated after pick-drawer.")
            return

        # ----- Step-5: pull(gripper, bottom)
        #   Pull the drawer 20 cm along +X (adjust if your scene uses a
        #   different slide direction).
        obs, reward, done = pull(env, task,
                                 pull_distance=0.20,
                                 pull_axis='x')
        if done:
            print("[Plan] Terminated after pull.")
            return

        # ----- Step-6/7: pick tomato1  →  place on plate
        obs, reward, done = pick(env, task, tomato1_pos)
        if done:
            print("[Plan] Terminated after picking tomato1.")
            return
        obs, reward, done = place(env, task, plate_pos)
        if done:
            print("[Plan] Terminated after placing tomato1.")
            return

        # ----- Step-8/9: pick tomato2  →  place on plate
        obs, reward, done = pick(env, task, tomato2_pos)
        if done:
            print("[Plan] Terminated after picking tomato2.")
            return
        obs, reward, done = place(env, task, plate_pos)
        if done:
            print("[Plan] Completed successfully – tomatoes placed!")
        else:
            print("[Plan] Finished all steps (done flag was False).")

    finally:
        shutdown_environment(env)
        print("=====  End of Task  =====")


if __name__ == "__main__":
    run_open_drawer_and_place_tomatoes()
