# ---------------------------------------------------------------------------
#  run_task_with_plan.py
#
#  Executes the fixed 9-step oracle plan that
#    1) opens the *bottom* drawer and
#    2) transfers the two tomatoes onto the plate,
#  leaving every other item untouched.
#
#  Only the already-implemented primitives contained in `skill_code`
#  (rotate, move, pull, pick, place) are invoked – no new skills are added.
#
#  The script is completely self-contained and may be launched as the
#  main entry point of the submission.
# ---------------------------------------------------------------------------

import numpy as np
from scipy.spatial.transform import Rotation as R

from env import setup_environment, shutdown_environment
from video import (
    init_video_writers,
    recording_step,
    recording_get_observation,
)
from object_positions import get_object_positions

# === Pre-implemented motion / manipulation primitives ===
from skill_code import rotate, move, pull, pick, place


def run_task_with_plan() -> None:
    """
    Drive the robot through the 9 oracle steps defined in the specification.

    Plan
    ----
    1) rotate          – align gripper 90° for a side approach
    2) move-to-side    – reach drawer’s predefined side waypoint
    3) move-to-anchor  – move straight to the drawer handle
    4) pick-drawer     – grasp the drawer handle (implemented via `pick`)
    5) pull            – pull the drawer open by 0.20 m
    6) pick tomato1    – pick first tomato from the table
    7) place tomato1   – place it onto the plate
    8) pick tomato2    – pick second tomato
    9) place tomato2   – place it onto the plate
    """

    print("\n================  RUN TASK – OPEN DRAWER & MOVE TOMATOES  ================")

    # ---------------------------------------------------------------------- #
    # 1)  Environment setup                                                  #
    # ---------------------------------------------------------------------- #
    env, task = setup_environment()
    try:
        # Reset the RLBench task
        descriptions, obs = task.reset()

        # Optional video capture (records every step if video helpers are active)
        init_video_writers(obs)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # ------------------------------------------------------------------ #
        # 2)  Retrieve all object positions we will need                     #
        # ------------------------------------------------------------------ #
        positions = get_object_positions()

        required_keys = [
            "bottom_side_pos",
            "bottom_anchor_pos",
            "tomato1",
            "tomato2",
            "plate",
        ]
        missing_keys = [k for k in required_keys if k not in positions]
        if missing_keys:
            raise KeyError(f"[run_task] Missing positions for {missing_keys}")

        # Convert positions into numpy arrays for downstream convenience
        side_pos = np.asarray(positions["bottom_side_pos"])
        anchor_pos = np.asarray(positions["bottom_anchor_pos"])
        tomato1_pos = np.asarray(positions["tomato1"])
        tomato2_pos = np.asarray(positions["tomato2"])
        plate_pos = np.asarray(positions["plate"])

        # ------------------------------------------------------------------ #
        # 3)  Execute the oracle plan (exact 9 steps from specification)     #
        # ------------------------------------------------------------------ #

        # ---- Step 1  ROTATE gripper from zero_deg to ninety_deg ----------- #
        print("\n[Step 1] rotate gripper → +90° around z-axis")
        current_quat = obs.gripper_pose[3:7]  # xyzw
        target_quat = (
            R.from_quat(current_quat) * R.from_euler("z", 90, degrees=True)
        ).as_quat()
        obs, reward, done = rotate(env, task, target_quat)
        if done:
            print("[run_task] Episode finished unexpectedly after rotate.")
            return

        # ---- Step 2  MOVE-TO-SIDE (bottom drawer side waypoint) ---------- #
        print("\n[Step 2] move gripper to bottom drawer side position")
        obs, reward, done = move(env, task, target_pos=side_pos)
        if done:
            print("[run_task] Episode finished unexpectedly after move-to-side.")
            return

        # ---- Step 3  MOVE-TO-ANCHOR (drawer handle) ---------------------- #
        print("\n[Step 3] move gripper to bottom drawer anchor (handle)")
        obs, reward, done = move(env, task, target_pos=anchor_pos)
        if done:
            print("[run_task] Episode finished unexpectedly after move-to-anchor.")
            return

        # ---- Step 4  PICK-DRAWER (grasp handle) -------------------------- #
        print("\n[Step 4] grasp drawer handle (pick-drawer)")
        obs, reward, done = pick(
            env,
            task,
            target_pos=anchor_pos,
            approach_distance=0.04,  # already at handle – only a tiny approach
            approach_axis="z",
        )
        if done:
            print("[run_task] Episode finished unexpectedly during pick-drawer.")
            return

        # ---- Step 5  PULL drawer open ------------------------------------ #
        print("\n[Step 5] pull drawer outwards by 0.20 m along +x")
        obs, reward, done = pull(
            env,
            task,
            pull_distance=0.20,
            pull_axis="x",
        )
        if done:
            print("[run_task] Episode finished unexpectedly during pull.")
            return

        # ---- Step 6  PICK tomato1 ---------------------------------------- #
        print("\n[Step 6] pick tomato1 from table")
        obs, reward, done = pick(
            env,
            task,
            target_pos=tomato1_pos,
            approach_distance=0.15,
            approach_axis="z",
        )
        if done:
            print("[run_task] Episode finished unexpectedly during pick tomato1.")
            return

        # ---- Step 7  PLACE tomato1 on plate ------------------------------ #
        print("\n[Step 7] place tomato1 onto plate")
        obs, reward, done = place(
            env,
            task,
            target_pos=plate_pos,
            approach_distance=0.15,
            approach_axis="z",
        )
        if done:
            print("[run_task] Episode finished unexpectedly during place tomato1.")
            return

        # ---- Step 8  PICK tomato2 ---------------------------------------- #
        print("\n[Step 8] pick tomato2 from table")
        obs, reward, done = pick(
            env,
            task,
            target_pos=tomato2_pos,
            approach_distance=0.15,
            approach_axis="z",
        )
        if done:
            print("[run_task] Episode finished unexpectedly during pick tomato2.")
            return

        # ---- Step 9  PLACE tomato2 on plate ------------------------------ #
        print("\n[Step 9] place tomato2 onto plate")
        obs, reward, done = place(
            env,
            task,
            target_pos=plate_pos,
            approach_distance=0.15,
            approach_axis="z",
        )

        # ------------------------------------------------------------------ #
        # 4)  Final outcome                                                  #
        # ------------------------------------------------------------------ #
        if done:
            print(
                "\n[run_task] SUCCESS – All plan steps completed, tomatoes placed!"
            )
        else:
            # RLBench tasks typically signal success automatically,
            # but if not, we still report completion of scripted actions.
            print(
                "\n[run_task] Plan executed. Verify success conditions in the environment."
            )

    finally:
        # Always shut down the simulation environment, even on failure
        shutdown_environment(env)
        print("================  RUN TASK – FINISHED  ================")


# Allow running directly
if __name__ == "__main__":
    run_task_with_plan()