import numpy as np
from scipy.spatial.transform import Rotation as SciR

from env import setup_environment, shutdown_environment
from object_positions import get_object_positions

# --- Skills (imported exactly as provided – DO NOT redefine) -----------------
from skill_code import move, rotate, pick, pull, place
# -----------------------------------------------------------------------------

from video import init_video_writers, recording_step, recording_get_observation


print("\n================  STARTING TASK EXECUTION  ================\n")

# -------------------------------------------------------------------------
#  Environment set-up
# -------------------------------------------------------------------------
env, task = setup_environment()

# Reset to the initial state
descriptions, obs = task.reset()

# ------------- Optional video writer (kept from skeleton) -------------
init_video_writers(obs)
task.step = recording_step(task.step)                 # wrap step call
task.get_observation = recording_get_observation(     # wrap get_obs
    task.get_observation
)

# ---------------------------------------------------------------------
#  Retrieve useful object positions
# ---------------------------------------------------------------------
positions = get_object_positions()

# Drawer (bottom) ------------------------------------------------------
side_pos    = np.asarray(positions['bottom_side_pos'])
anchor_pos  = np.asarray(positions['bottom_anchor_pos'])  # handle

# 90 deg rotation around local-Z
target_quat = SciR.from_euler('z', 90, degrees=True).as_quat()

# (Positive X is assumed to be the pull-out direction; adjust if needed)
pull_distance = 0.15
pull_axis     = 'x'

# Tomatoes & goal plate ----------------------------------------------
tomatoes  = ['item1', 'item2']            # two tomatoes to move
plate_pos = np.asarray(positions['plate'])


# ---------------------------------------------------------------------
#  Oracle Plan  (specification steps 1-9)
#  NOTE: The next five lines are FROZEN – DO NOT edit.
# ---------------------------------------------------------------------
obs, reward, done = move(env, task, side_pos)
obs, reward, done = rotate(env, task, target_quat)
obs, reward, done = move(env, task, anchor_pos)
obs, reward, done = pick(env, task, anchor_pos, approach_axis='y')
obs, reward, done = pull(env, task, pull_distance, pull_axis=pull_axis)

# ---------------------------------------------------------------------
#  Steps 6-9 – move each tomato onto the plate
# ---------------------------------------------------------------------
if not done:
    for tomato in tomatoes:
        print(f"\n---- Picking {tomato} ----")
        tomato_pos = np.asarray(positions[tomato])
        obs, reward, done = pick(env, task, tomato_pos)
        if done:
            break

        print(f"\n---- Placing {tomato} on plate ----")
        obs, reward, done = place(env, task, plate_pos)
        if done:
            break

# ---------------------------------------------------------------------
#  Task completed / shut down
# ---------------------------------------------------------------------
if done:
    print("\n[Task] Task finished with done=True (possibly early termination).")
else:
    print("\n================  TASK COMPLETED SUCCESSFULLY  ================\n")

shutdown_environment(env)