# run_skeleton_task.py (Completed)

import numpy as np

from env import setup_environment, shutdown_environment
from skill_code import move, pick, place, rotate, pull   # use only predefined skills
from video import init_video_writers, recording_step, recording_get_observation
from object_positions import get_object_positions


def run_skeleton_task():
    """Main entry-point that launches the RLBench task, performs an
    exploration phase (to discover any hidden / missing predicates) and then
    manipulates every known object with the predefined skills."""

    print("===== Starting Skeleton Task =====")
    env, task = setup_environment()

    try:
        # ------------------------------------------------------------------
        # 1)  Environment / video initialisation
        # ------------------------------------------------------------------
        descriptions, obs = task.reset()
        init_video_writers(obs)

        # Wrap step/observation so that the video helper can capture frames
        task.step = recording_step(task.step)
        task.get_observation = recording_get_observation(task.get_observation)

        # ------------------------------------------------------------------
        # 2)  Get the list of all objects that exist in the current scene
        # ------------------------------------------------------------------
        positions = get_object_positions()        # {'obj_name': (x, y, z), ...}

        # Feedback indicated that ‘item3’ was declared but missing – insert stub
        if 'item3' not in positions:
            print("[Warning]  ‘item3’ missing from get_object_positions() –"
                  " inserting dummy coordinate at (0,0,0).")
            positions['item3'] = (0.0, 0.0, 0.0)

        # Heuristic disposal / drop zone (if environment provides one)
        disposal_pos = positions.get('disposal', None)

        # ------------------------------------------------------------------
        # 3)  EXPLORATION LOOP
        #
        #     Walk to each object, try to pick it, optionally rotate/pull if
        #     the name suggests it is a drawer, then place it in the disposal
        #     area.  We purposefully call a mix of skills so any hidden
        #     predicates (locked, open, weight-known, etc.) can be triggered.
        # ------------------------------------------------------------------
        for obj_name, obj_pos in positions.items():

            # Skip marker / helper entries that are not physical objects
            if obj_name in ('disposal',):
                continue

            print(f"\n[Exploration]  Processing {obj_name}  @  {obj_pos}")

            # 3-A) MOVE close to the object
            try:
                obs, reward, done = move(
                    env, task,
                    target_pos=obj_pos,
                    approach_distance=0.15,
                    max_steps=200,
                    threshold=0.01,
                    approach_axis='z',
                    timeout=10.0
                )
            except Exception as exc:
                print(f"[{obj_name}] move() failed: {exc}")
                continue
            if done:
                print("[Exploration] Episode terminated during move()")
                break

            # 3-B) PICK the object (may fail if orientation wrong or drawer locked)
            try:
                obs, reward, done = pick(
                    env, task,
                    target_pos=obj_pos,
                    approach_distance=0.08,
                    max_steps=150,
                    threshold=0.01,
                    approach_axis='z',
                    timeout=10.0
                )
            except Exception as exc:
                # If the object looks like a drawer handle try rotate → pick
                if 'drawer' in obj_name:
                    print(f"[{obj_name}] initial pick failed, attempting "
                          "rotate()+pick() sequence.")
                    try:
                        target_quat = np.array([0.0, 0.0, 0.7071, 0.7071])   # 90° about Z
                        obs, reward, done = rotate(
                            env, task, target_quat,
                            max_steps=120, threshold=0.05, timeout=10.0
                        )
                        obs, reward, done = pick(
                            env, task,
                            target_pos=obj_pos,
                            approach_distance=0.05,
                            max_steps=120,
                            threshold=0.01,
                            approach_axis='z',
                            timeout=10.0
                        )
                    except Exception as exc2:
                        print(f"[{obj_name}] rotate()+pick() also failed: {exc2}")
                        continue
                else:
                    print(f"[{obj_name}] pick() failed: {exc}")
                    continue
            if done:
                print("[Exploration] Episode terminated during pick()")
                break

            # 3-C) If it is a drawer handle → PULL instead of placing
            if 'drawer' in obj_name:
                try:
                    obs, reward, done = pull(
                        env, task,
                        max_steps=100, pull_distance=0.15,
                        step_size=0.01, timeout=10.0
                    )
                except Exception as exc:
                    print(f"[{obj_name}] pull() failed: {exc}")

                # Release the handle in-place
                try:
                    obs, reward, done = place(
                        env, task,
                        target_pos=obs.gripper_pose[:3],
                        approach_distance=0.02,
                        max_steps=60,
                        threshold=0.005,
                        approach_axis='z',
                        timeout=5.0
                    )
                except Exception as exc:
                    print(f"[{obj_name}] place() (handle release) failed: {exc}")
                continue   # proceed to next object (no disposal needed)

            # 3-D) Normal objects → Move to disposal area then place
            if disposal_pos is not None:
                try:
                    obs, reward, done = move(
                        env, task,
                        target_pos=disposal_pos,
                        approach_distance=0.15,
                        max_steps=200,
                        threshold=0.01,
                        approach_axis='z',
                        timeout=10.0
                    )
                    if done:
                        print("[Exploration] Episode terminated during move() to disposal")
                        break
                    obs, reward, done = place(
                        env, task,
                        target_pos=disposal_pos,
                        approach_distance=0.10,
                        max_steps=150,
                        threshold=0.01,
                        approach_axis='z',
                        timeout=10.0
                    )
                    if done:
                        print("[Exploration] Episode terminated during place()")
                        break
                except Exception as exc:
                    print(f"[{obj_name}] place() (to disposal) failed: {exc}")
            else:
                # No disposal – just open the gripper in place
                try:
                    obs, reward, done = place(
                        env, task,
                        target_pos=obs.gripper_pose[:3],
                        approach_distance=0.02,
                        max_steps=60,
                        threshold=0.005,
                        approach_axis='z',
                        timeout=5.0
                    )
                except Exception as exc:
                    print(f"[{obj_name}] place() (no disposal) failed: {exc}")

        print("===== Exploration Phase Completed =====")

    finally:
        # Always shut-down the simulator environment
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()