import re
import time
import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

# ----- Skills (already supplied in skill_code.py) -----
from skill_code import rotate, pick, place, move, pull

from video import init_video_writers, recording_step, recording_get_observation
from object_positions import get_object_positions


# ------------------------------------------------------------------
#  Utility:  Parse a PDDL domain string and return the set of predicates
# ------------------------------------------------------------------
_PRED_RE = re.compile(r'\(\s*([a-zA-Z0-9_\-]+)\s')

def _extract_predicates(domain_pddl: str):
    """Return a set with all predicate names occurring in a (:predicates …) block."""
    preds = set()
    # Locate the (:predicates …) region(s)
    segments = domain_pddl.split('(:predicates')
    for seg in segments[1:]:
        # scan until matching ')' that balances the open
        depth = 1
        block = ''
        for ch in seg:
            if ch == '(':
                depth += 1
            elif ch == ')':
                depth -= 1
                if depth == 0:
                    break
            block += ch
        # within the block, extract words that look like “(name”
        for match in _PRED_RE.finditer(block):
            # first match in each group is the predicate name
            preds.add(match.group(1))
    return preds


# ------------------------------------------------------------------
# The two domain strings (combined-domain and exploration)
# ------------------------------------------------------------------
COMBINED_DOMAIN = """(define (domain combined-domain)
  (:requirements :strips :typing :negative-preconditions :equality :disjunctive-preconditions)

  (:types
    object
    location
    drawer - object
    gripper - object
    position - object
    angle - object
  )

  (:predicates
    (at ?obj - object ?loc - location)
    (holding ?obj - object)
    (handempty)

    (is-locked ?d - drawer)
    (is-open ?d - drawer)
    (rotated ?g - gripper ?a - angle)
    (gripper-at ?g - gripper ?p - position)
    (holding-drawer ?g - gripper ?d - drawer)
    (is-side-pos ?p - position ?d - drawer)
    (is-anchor-pos ?p - position ?d - drawer)
  )

  (:action pick
    :parameters (?obj - object ?loc - location)
    :precondition (and
      (at ?obj ?loc)
      (handempty)
    )
    :effect (and
      (holding ?obj)
      (not (handempty))
      (not (at ?obj ?loc))
    )
  )

  (:action place
    :parameters (?obj - object ?loc - location)
    :precondition (and
      (holding ?obj)
    )
    :effect (and
      (at ?obj ?loc)
      (handempty)
      (not (holding ?obj))
    )
  )

  (:action rotate
    :parameters (?g - gripper ?from - angle ?to - angle)
    :precondition (and
      (rotated ?g ?from)
      (not (= ?from ?to))
    )
    :effect (and
      (not (rotated ?g ?from))
      (rotated ?g ?to)
    )
  )

  (:action move-to-side
    :parameters (?g - gripper ?d - drawer ?from ?to - position)
    :precondition (and
      (gripper-at ?g ?from)
      (is-side-pos ?to ?d)
      (not (gripper-at ?g ?to))
      (rotated ?g ninety_deg)
    )
    :effect (and
      (gripper-at ?g ?to)
      (not (gripper-at ?g ?from))
    )
  )

  (:action move-to-anchor
    :parameters (?g - gripper ?d - drawer ?from ?to - position)
    :precondition (and
      (is-side-pos ?from ?d)
      (is-anchor-pos ?to ?d)
      (gripper-at ?g ?from)
      (not (gripper-at ?g ?to))
    )
    :effect (and
      (gripper-at ?g ?to)
      (not (gripper-at ?g ?from))
    )
  )

  (:action pick-drawer
    :parameters (?g - gripper ?d - drawer ?p - position)
    :precondition (and
      (gripper-at ?g ?p)
      (is-anchor-pos ?p ?d)
      (not (holding-drawer ?g ?d))
    )
    :effect (holding-drawer ?g ?d)
  )

  (:action pull
    :parameters (?g - gripper ?d - drawer)
    :precondition (and
      (holding-drawer ?g ?d)
      (not (is-locked ?d))
      (not (is-open ?d))
    )
    :effect (is-open ?d)
  )
)
"""

EXPLORATION_DOMAIN = """(define (domain exploration)
  (:requirements :strips :typing :conditional-effects :universal-preconditions)
  (:types 
    robot object location
  )
  
  (:predicates
    (robot-at ?r - robot ?loc - location)
    (at ?obj - object ?loc - location)
    (identified ?obj - object)
    (temperature-known ?obj - object)
    (holding ?obj - object)
    (handempty)
    (weight-known ?obj - object)
    (durability-known ?obj - object)
  )
  
  (:action move
    :parameters (?r - robot ?from - location ?to - location)
    :precondition (robot-at ?r ?from)
    :effect (and
      (not (robot-at ?r ?from))
      (robot-at ?r ?to)
      (forall (?obj - object)
        (when (at ?obj ?to)
          (identified ?obj)
        )
      )
    )
  )
  
  (:action move
    :parameters (?r - robot ?from - location ?to - location)
    :precondition (robot-at ?r ?from)
    :effect (and
      (not (robot-at ?r ?from))
      (robot-at ?r ?to)
      (forall (?obj - object)
        (when (at ?obj ?to)
          (temperature-known ?obj)
        )
      )
    )
  )
  
  (:action pick
    :parameters (?r - robot ?obj - object ?loc - location)
    :precondition (and
       (robot-at ?r ?loc)
       (at ?obj ?loc)
       (handempty)
    )
    :effect (and
      (holding ?obj)
      (not (handempty))
      (not (at ?obj ?loc))
      (weight-known ?obj)
    )
  )
  
  (:action pick
    :parameters (?r - robot ?obj - object ?loc - location)
    :precondition (and
       (robot-at ?r ?loc)
       (at ?obj ?loc)
       (handempty)
    )
    :effect (and
      (holding ?obj)
      (not (handempty))
      (not (at ?obj ?loc))
      (durability-known ?obj)
    )
  )

  (:action pull
    :parameters (?r - robot ?obj - object ?loc - location)
    :precondition (and
       (robot-at ?r ?loc)
       (at ?obj ?loc)
       (holding ?obj)
       (not (lock-known ?obj))
    )
    :effect (lock-known ?obj)
  )
)
"""


def run_skeleton_task():
    """Main entry: set-up the simulated environment, perform a lightweight
    ‘exploration’ to locate predicates that exist in the exploration domain
    but not in the working (combined) domain, and finally shut down."""
    print("===== Starting Skeleton Task =====")

    # ---------------------------------------------------
    # 1)  Spin up the RLBench environment & task
    # ---------------------------------------------------
    env, task = setup_environment()

    try:
        # Reset task so that we have a valid, clean starting point
        descriptions, obs = task.reset()

        # Optional: enable video recording helper wrappers
        init_video_writers(obs)
        task.step = recording_step(task.step)
        task.get_observation = recording_get_observation(task.get_observation)

        # ---------------------------------------------------
        # 2)  Determine the predicate mismatch
        # ---------------------------------------------------
        combined_preds     = _extract_predicates(COMBINED_DOMAIN)
        exploration_preds  = _extract_predicates(EXPLORATION_DOMAIN)

        missing_predicates = sorted(list(exploration_preds - combined_preds))

        print("\n[Exploration] Predicates present in EXPLORATION domain but"
              " missing from COMBINED domain:")
        for p in missing_predicates:
            print("   •", p)

        # A very concise “use” of at least one available skill so that the
        # program interacts with the robot in a safe, no-op fashion.
        #
        # We command a zero-rotation of the gripper; this is harmless and
        # exercises the `rotate` skill implementation without changing pose.
        #
        # Rationale:  ‑ Keep within the requirement of “only use predefined
        #              skills”, yet do something that cannot fail regardless
        #              of the concrete task (no geometry assumptions needed).
        #
        try:
            identity_quat = np.array([0., 0., 0., 1.])  # xyzw
            # Call rotate once; if the skill requires a certain angle
            # difference it will early-exit immediately.
            rotate(env, task, identity_quat,
                   max_steps=1,     # one iteration is enough for a no-op
                   threshold=0.0,
                   timeout=1.0)
        except Exception as exc:
            # The call is best-effort; any unexpected error is logged but
            # does not abort the overall exploration & shutdown sequence.
            print("[Exploration] Warning: rotate skill call failed –", exc)

        # ---------------------------------------------------
        # 3)  Finished – nothing more to do for this analysis script
        # ---------------------------------------------------
        print("\n[Exploration] Completed predicate-discovery phase.")

    finally:
        shutdown_environment(env)   # always clean-up

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()