# run_skeleton_task.py   (completed)

import time
import traceback
import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment
from skill_code import *          # → contains pick, place, move, rotate, pull
from video import (init_video_writers,
                   recording_step,
                   recording_get_observation)
from object_positions import get_object_positions


# ------------------------------------------------------------------------------|
# Helper: very light-weight PDDL predicate parser (only “:predicates” section)   |
# ------------------------------------------------------------------------------|
def _parse_predicates_from_pddl(pddl_str: str):
    preds = set()
    inside = False
    for line in pddl_str.splitlines():
        line = line.strip()
        if not inside:
            if line.lower().startswith("(:predicates"):
                inside = True
            continue
        # terminate at the next “(:” that is NOT predicates.
        if line.startswith("(:") and not line.lower().startswith("(:predicates"):
            break
        if not line or line.startswith(";"):
            continue
        # tokens like “(at ?o - object ?l - location)”
        if line.startswith("("):
            pred_name = line.split()[0][1:]      # strip leading '('
            preds.add(pred_name)
    return preds


# ------------------------------------------------------------------------------|
# The combined domain PDDL (truncated after the predicates section)             |
# ------------------------------------------------------------------------------|
_COMBINED_DOMAIN_PDDL = """
(define (domain combined-domain)
 (:requirements :strips :typing :negative-preconditions :equality :disjunctive-preconditions)
 (:types object location drawer - object gripper - object position - object angle - object)
 (:predicates
   (at ?obj - object ?loc - location)
   (holding ?obj - object)
   (handempty)
   (is-locked ?d - drawer)
   (is-open ?d - drawer)
   (rotated ?g - gripper ?a - angle)
   (gripper-at ?g - gripper ?p - position)
   (holding-drawer ?g - gripper ?d - drawer)
   (is-side-pos ?p - position ?d - drawer)
   (is-anchor-pos ?p - position ?d - drawer)
 )
)
"""

_EXPLORATION_PDDL = """
(define (domain exploration)
 (:requirements :strips :typing :conditional-effects :universal-preconditions)
 (:types robot object location)
 (:predicates
   (robot-at ?r - robot ?loc - location)
   (at ?obj - object ?loc - location)
   (identified ?obj - object)
   (temperature-known ?obj - object)
   (holding ?obj - object)
   (handempty)
   (weight-known ?obj - object)
   (durability-known ?obj - object)
 )
)
"""


def _explore_and_find_missing_predicate():
    """
    Simple static “exploration” – compare the exploration-domain predicates
    with the combined task domain predicates and output what is missing.
    """
    combined_preds = _parse_predicates_from_pddl(_COMBINED_DOMAIN_PDDL)
    exploration_preds = _parse_predicates_from_pddl(_EXPLORATION_PDDL)

    missing = sorted(list(exploration_preds - combined_preds))
    if missing:
        print("\n[Exploration] Detected predicates present in exploration domain "
              "but missing in the task domain:")
        for p in missing:
            print(f"   • {p}")
    else:
        print("\n[Exploration] No missing predicates detected!")

    return missing


# ------------------------------------------------------------------------------|
# Main routine                                                                  |
# ------------------------------------------------------------------------------|
def run_skeleton_task():
    """Generic skeleton for running any task in your simulation."""
    print("===== Starting Skeleton Task =====")

    # ------------------------------------------------------------------|
    # (1) Environment setup                                             |
    # ------------------------------------------------------------------|
    env, task = setup_environment()
    try:
        # Reset the task
        descriptions, obs = task.reset()

        # Optional: start video recording
        init_video_writers(obs)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # ------------------------------------------------------------------|
        # (2) “Exploration” – discover missing predicate(s)                 |
        # ------------------------------------------------------------------|
        missing_preds = _explore_and_find_missing_predicate()

        # If ‘handempty’ is missing (feedback told us so), we ensure that
        # our run ends in a state where the gripper is empty, by performing
        # a simple pick-and-place sequence.
        need_handempty = "handempty" in missing_preds
        if need_handempty:
            print("\n[Task] Will demonstrate how to satisfy handempty "
                  "by performing pick → place sequence.")

        # ------------------------------------------------------------------|
        # (3) Retrieve object positions for a very small demo plan          |
        # ------------------------------------------------------------------|
        try:
            positions = get_object_positions()
        except Exception as e:
            positions = {}
            print("[Warning] get_object_positions() failed:", e)

        if not positions:
            print("[Task] No object positions available – nothing to do.")
            return

        # Select the first object arbitrarily
        obj_name, obj_pos = next(iter(positions.items()))
        print(f"\n[Task] Selected object '{obj_name}' at position {obj_pos}")

        # ------------------------------------------------------------------|
        # (4) Execute a tiny oracle-like plan: pick then place              |
        # ------------------------------------------------------------------|
        try:
            print(f"[Skill: pick] Attempting to pick '{obj_name}' …")
            obs, reward, done = pick(env,
                                      task,
                                      target_pos=np.array(obj_pos),
                                      approach_distance=0.15,
                                      max_steps=150,
                                      threshold=0.005,
                                      approach_axis='z',
                                      timeout=15.0)
            if done:
                print("[Task] Task ended during pick – terminating early.")
                return
        except Exception as e:
            print("[Error] pick() failed:")
            traceback.print_exc()
            return

        # Compute a nearby place position (offset on +x by 0.12 m)
        place_pos = np.array(obj_pos) + np.array([0.12, 0.0, 0.0])
        print(f"[Task] Computed place position: {place_pos}")

        try:
            print(f"[Skill: place] Placing '{obj_name}' …")
            obs, reward, done = place(env,
                                       task,
                                       target_pos=place_pos,
                                       approach_distance=0.15,
                                       max_steps=150,
                                       threshold=0.005,
                                       approach_axis='z',
                                       timeout=15.0)
            if done:
                print("[Task] Task ended during place – terminating early.")
                return
        except Exception as e:
            print("[Error] place() failed:")
            traceback.print_exc()
            return

        # ------------------------------------------------------------------|
        # (5) Final message / predicate confirmation                        |
        # ------------------------------------------------------------------|
        print("\n===== Demo plan finished. =====")
        if need_handempty:
            print("[Result] The robot ended with an open gripper – "
                  "predicate ‘handempty’ should now hold true.")
        else:
            print("[Result] Plan executed without explicit handempty "
                  "requirement.")

    finally:
        # Always cleanly shut down the environment
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()
