# run_skeleton_task.py (Completed)

import re
import time
import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment
from skill_code import rotate, move, pick, place, pull          # do NOT redefine – just import
from video import init_video_writers, recording_step, recording_get_observation
from object_positions import get_object_positions


# ---------------------------------------------------------------------------
# 1)  DOMAIN SOURCE (for static text-parsing exploration)
# ---------------------------------------------------------------------------
DOMAIN_PDDL = """
(define (domain combined-domain)
  (:requirements :strips :typing :negative-preconditions :equality :disjunctive-preconditions)

  (:types
    object
    location
    drawer - object
    gripper - object
    position - object
    angle - object
  )

  (:predicates
    (at ?obj - object ?loc - location)
    (holding ?obj - object)
    (handempty)
    (is-locked ?d - drawer)
    (is-open ?d - drawer)
    (rotated ?g - gripper ?a - angle)
    (gripper-at ?g - gripper ?p - position)
    (holding-drawer ?g - gripper ?d - drawer)
    (is-side-pos ?p - position ?d - drawer)
    (is-anchor-pos ?p - position ?d - drawer)
  )

  (:action pick
    :parameters (?obj - object ?loc - location)
    :precondition (and (at ?obj ?loc) (handempty))
    :effect (and (holding ?obj) (not (handempty)) (not (at ?obj ?loc)))
  )

  (:action place
    :parameters (?obj - object ?loc - location)
    :precondition (and (holding ?obj))
    :effect (and (at ?obj ?loc) (handempty) (not (holding ?obj)))
  )

  (:action rotate
    :parameters (?g - gripper ?from - angle ?to - angle)
    :precondition (and (rotated ?g ?from) (not (= ?from ?to)))
    :effect (and (not (rotated ?g ?from)) (rotated ?g ?to))
  )

  (:action move-to-side
    :parameters (?g - gripper ?d - drawer ?from ?to - position)
    :precondition (and (gripper-at ?g ?from) (is-side-pos ?to ?d) (not (gripper-at ?g ?to)) (rotated ?g ninety_deg))
    :effect (and (gripper-at ?g ?to) (not (gripper-at ?g ?from)))
  )

  (:action move-to-anchor
    :parameters (?g - gripper ?d - drawer ?from ?to - position)
    :precondition (and (is-side-pos ?from ?d) (is-anchor-pos ?to ?d) (gripper-at ?g ?from) (not (gripper-at ?g ?to)))
    :effect (and (gripper-at ?g ?to) (not (gripper-at ?g ?from)))
  )

  (:action pick-drawer
    :parameters (?g - gripper ?d - drawer ?p - position)
    :precondition (and (gripper-at ?g ?p) (is-anchor-pos ?p ?d) (not (holding-drawer ?g ?d)))
    :effect (holding-drawer ?g ?d)
  )

  (:action pull
    :parameters (?g - gripper ?d - drawer)
    :precondition (and (holding-drawer ?g ?d) (not (is-locked ?d)) (not (is-open ?d)))
    :effect (is-open ?d)
  )
)
"""

# ---------------------------------------------------------------------------
# 2)  HELPER – Simple PDDL parser to discover missing predicates
# ---------------------------------------------------------------------------
def discover_missing_predicates(domain_text: str):
    """
    Parse the given PDDL domain to find predicates that are used
    in preconditions / effects but never declared in the (:predicates ...) block.
    Returns a set with the missing predicate names.
    """
    # 2.1  Pull the declared predicate names
    declared_block = re.search(r'\(:predicates(.*?)\)', domain_text, re.DOTALL)
    declared_predicates = set()
    if declared_block:
        for line in declared_block.group(1).splitlines():
            line = line.strip()
            if not line or line.startswith(';'):
                continue
            m = re.match(r'\(([^ )]+)', line)          # first symbol inside "("
            if m:
                declared_predicates.add(m.group(1))

    # 2.2  Scan the rest of the file for all predicate mentions
    tokens = re.findall(r'\([^() ]+', domain_text)     # all "(" followed by token
    used_predicates = set(tok[1:] for tok in tokens
                          if tok[1:] not in [':action', ':parameters', ':precondition',
                                            ':effect', 'and', 'not', '=', 'forall',
                                            'when', 'exists'])
    # 2.3  Missing = used-but-not-declared
    return used_predicates - declared_predicates


# ---------------------------------------------------------------------------
# 3)  MAIN EXECUTION LOGIC
# ---------------------------------------------------------------------------
def run_skeleton_task():
    """Generic skeleton runner with a small exploration to find missing predicate(s)."""
    print("===== Starting Skeleton Task =====")

    # ============================================================
    # A)  STATIC EXPLORATION STAGE – Find undefined predicates
    # ============================================================
    print("\n[Exploration] Scanning PDDL domain for undefined predicates …")
    missing_preds = discover_missing_predicates(DOMAIN_PDDL)
    if missing_preds:
        print(f"[Exploration] Detected potential undefined predicates: {sorted(missing_preds)}")
    else:
        print("[Exploration] No undefined predicates detected in the given PDDL.")

    # ============================================================
    # B)  SIMULATION STAGE
    # ============================================================
    # ── Setup RLBench simulation environment
    env, task = setup_environment()
    try:
        descriptions, obs = task.reset()
        init_video_writers(obs)

        # Wrap step / observation to enable automatic recording
        task.step = recording_step(task.step)
        task.get_observation = recording_get_observation(task.get_observation)

        # --------------------------------------------------------
        # Retrieve positions that may be required for skills
        # --------------------------------------------------------
        positions = get_object_positions()
        print(f"[Info] Object positions from helper: {positions}")

        # --------------------------------------------------------
        # VERY GENERIC DEMO PLAN
        # --------------------------------------------------------
        # Note: Without a concrete task description, we demonstrate
        #       one of the available skills (rotate) on the gripper,
        #       keeping the rest of the skeleton intact.
        #
        #       A real solution would compose pick/move/place/pull
        #       calls according to an oracle plan.
        #
        print("\n[Demo] Performing a small rotate-in-place to show skill usage.")
        identity_quat = np.array([0.0, 0.0, 0.0, 1.0])  # xyzw
        obs, reward, done = rotate(
            env,
            task,
            target_quat=identity_quat,
            max_steps=60,
            threshold=0.03,
            timeout=5.0
        )
        if done:
            print("[Demo] Task completed (simulation reported done).")

        # --------------------------------------------------------
        # Placeholder for further oracle plan steps …
        # (pick, move, place, pull, etc.)
        # --------------------------------------------------------

    except Exception as e:
        print(f"[Error] An exception occurred during task execution: {e}")

    finally:
        # Always cleanly shut down the environment
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()
